package dev.fitko.fitconnect.core.validation.xml;

import java.io.ByteArrayInputStream;
import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public final class SchemaSystemIdExtractor {

    private static final String XSI_NS = XMLConstants.W3C_XML_SCHEMA_INSTANCE_NS_URI;
    private static final String ATTR_SCHEMA_LOCATION = "schemaLocation";
    private static final String ATTR_NO_NAMESPACE_SCHEMA_LOCATION = "noNamespaceSchemaLocation";

    private SchemaSystemIdExtractor() {}

    /**
     * Extracts the systemId from an XML document.
     * <p>
     * Order:
     * <ol>
     *   <li>{@code xsi:schemaLocation} (root element namespace)</li>
     *   <li>{@code xsi:noNamespaceSchemaLocation}</li>
     * </ol>
     * <p>
     * Returns an empty string if nothing was resolved or if a parsing error occurred.
     */
    public static String extractSystemId(byte[] xmlBytes) {
        if (xmlBytes == null || xmlBytes.length == 0) {
            return "";
        }

        try {
            Document document = parseXml(xmlBytes);
            Element root = document.getDocumentElement();
            if (root == null) {
                return "";
            }

            String rootNamespace = trim(root.getNamespaceURI());
            String systemId = extractFromSchemaLocation(root, rootNamespace);
            if (systemId != null) {
                return systemId;
            }

            return extractFromNoNamespaceSchemaLocation(root);
        } catch (Exception e) {
            return "";
        }
    }

    private static Document parseXml(byte[] xmlBytes) throws Exception {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setNamespaceAware(true);
        factory.setFeature(XMLConstants.FEATURE_SECURE_PROCESSING, true);
        factory.setFeature("http://apache.org/xml/features/disallow-doctype-decl", true);
        factory.setFeature("http://xml.org/sax/features/external-general-entities", false);
        factory.setFeature("http://xml.org/sax/features/external-parameter-entities", false);

        DocumentBuilder builder = factory.newDocumentBuilder();
        try (ByteArrayInputStream inputStream = new ByteArrayInputStream(xmlBytes)) {
            return builder.parse(inputStream);
        }
    }

    private static String extractFromNoNamespaceSchemaLocation(Element root) {
        String value = root.getAttributeNS(XSI_NS, ATTR_NO_NAMESPACE_SCHEMA_LOCATION);
        final String trimmed = trim(value);
        return trimmed == null ? "" : trimmed;
    }

    private static String extractFromSchemaLocation(Element root, String rootNamespace) {
        String schemaLocation = trim(root.getAttributeNS(XSI_NS, ATTR_SCHEMA_LOCATION));
        if (schemaLocation == null) {
            return null;
        }

        String[] tokens = schemaLocation.split("\\s+");
        if (tokens.length < 2) {
            // Less than one (namespace, location) pair
            return null;
        }

        String firstSystemId = null;

        // namespace URI + systemId pairs
        for (int i = 0; i + 1 < tokens.length; i += 2) {
            String ns = trim(tokens[i]);
            String loc = trim(tokens[i + 1]);

            if (loc == null) {
                continue; // skip incomplete pair
            }

            if (firstSystemId == null) {
                firstSystemId = loc; // fallback
            }

            if (rootNamespace != null && rootNamespace.equals(ns)) {
                return loc; // namespace uri match
            }
        }

        return firstSystemId;
    }

    private static String trim(String value) {
        if (value == null) {
            return null;
        }
        String trimmed = value.trim();
        return trimmed.isEmpty() ? null : trimmed;
    }
}
