package dev.fitko.fitconnect.core.validation.virusscan.process;

import dev.fitko.fitconnect.api.domain.validation.VirusScanResult;
import dev.fitko.fitconnect.core.io.ProcessRunner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Parser for ClamAV process scanner results.
 *
 * <p>This parser handles the various outputs from the clamscan command:
 *
 * <ul>
 *   <li>Clean files: "filename: OK"
 *   <li>Infected files: "filename: MalwareName FOUND"
 *   <li>Errors: "filename: Error message"
 *   <li>Statistics: "----------- SCAN SUMMARY -----------"
 * </ul>
 */
public class ClamAVProcessResultParser {

    private static final Logger LOGGER = LoggerFactory.getLogger(ClamAVProcessResultParser.class);

    // Match virus detection: "filename: MalwareName FOUND"
    private static final Pattern VIRUS_PATTERN = Pattern.compile("^(.+):\\s+(.+)\\s+FOUND$");

    // Match clean files: "filename: OK"
    private static final Pattern CLEAN_PATTERN = Pattern.compile("^(.+):\\s+OK$");

    // Match errors: "filename: Error message"
    private static final Pattern ERROR_PATTERN = Pattern.compile("^(.+):\\s+(.+)$");

    // Match summary section
    private static final Pattern SUMMARY_PATTERN = Pattern.compile("^----------- SCAN SUMMARY -----------$");

    /**
     * Parses the result from a ClamAV process execution.
     *
     * @param processResult the result from scanner
     * @return a VirusScanResult representing the scan outcome
     */
    public VirusScanResult parseResult(ProcessRunner.Result processResult) {
        if (processResult == null) {
            LOGGER.error("Process result is null");
            return VirusScanResult.ofScanFailed("Process result is null");
        }

        if (processResult.exitCode == -1) {
            String errorMessage = "ClamAV process failed to execute due to: " + processResult.error;
            LOGGER.error(errorMessage);
            return VirusScanResult.ofScanFailed(errorMessage);
        }

        String output = processResult.output;
        if (output == null || output.trim().isEmpty()) {
            LOGGER.warn("ClamAV process returned empty output");
            return VirusScanResult.ofScanFailed("Empty output from ClamAV process");
        }

        LOGGER.debug("Parsing ClamAV output");
        return parseOutput(output);
    }

    /**
     * Parses the ClamAV output to determine if malware was detected or a scan error occurred.
     *
     * <p>This method processes the output line by line, skipping the summary section. It returns
     * immediately when an infected file or scan error is found. If no issues are detected, it returns
     * the last clean result found.
     *
     * @param output the raw output from clamscan
     * @return a VirusScanResult
     */
    private VirusScanResult parseOutput(String output) {
        VirusScanResult currentResult = null;
        for (String rawLine : output.split("\\r?\\n")) {
            String line = rawLine.trim();
            if (!line.isEmpty()) {
                // ignore summary section
                if (SUMMARY_PATTERN.matcher(line).matches()) {
                    break;
                }
                currentResult = parseLine(line);
                if (currentResult.isInfected() || currentResult.isScanFailed()) {
                    return currentResult;
                }
            }
        }
        // the current result must be ok then
        return currentResult;
    }

    /**
     * Parses a single line from ClamAV output.
     *
     * @param line the line to parse
     * @return a VirusScanResult containing the parsed information
     */
    private VirusScanResult parseLine(String line) {
        // Check for detected viruses
        Matcher virusMatcher = VIRUS_PATTERN.matcher(line);
        if (virusMatcher.matches()) {
            String filename = virusMatcher.group(1);
            String signature = virusMatcher.group(2);
            LOGGER.warn("Malware detected in {}: {}", filename, signature);
            return VirusScanResult.ofInfected("Malware detected in " + filename, signature);
        }

        // Check for clean files
        Matcher cleanMatcher = CLEAN_PATTERN.matcher(line);
        if (cleanMatcher.matches()) {
            String filename = cleanMatcher.group(1);
            LOGGER.debug("File {} is clean", filename);
            return VirusScanResult.ofClean(line);
        }

        // Check for errors
        Matcher errorMatcher = ERROR_PATTERN.matcher(line);
        if (errorMatcher.matches()) {
            String filename = errorMatcher.group(1);
            String errorMessage = errorMatcher.group(2);
            LOGGER.error("ClamAV error for {}: {}", filename, errorMessage);
            return VirusScanResult.ofScanFailed("ClamAV error: " + errorMessage);
        }

        return VirusScanResult.ofScanFailed("ClamAV result output did not contain any matching data");
    }
}
