package dev.fitko.fitconnect.core.validation.virusscan.deamon.response;

import static dev.fitko.fitconnect.core.validation.virusscan.deamon.response.ClamAVStreamResponse.*;

import dev.fitko.fitconnect.api.domain.validation.VirusScanResult;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClamAVDeamonResultParser {

    private static final Logger LOGGER = LoggerFactory.getLogger(ClamAVDeamonResultParser.class);

    /**
     * Parses ClamAV response and returns a result.
     *
     * <p>Generic response parser that handles various ClamAV response formats:
     *
     * <ul>
     *   <li><code>stream: OK</code> - No virus found
     *   <li><code>stream: &lt;virus_name&gt; FOUND</code> - Virus detected
     *   <li><code>stream: ERROR</code> - Scan error occurred
     * </ul>
     *
     * @param responseData raw response from stream
     * @return Parsed {@link VirusScanResult}
     * @throws IOException if reading response fails
     */
    public VirusScanResult parseClamAVResponse(InputStream responseData) throws IOException {
        final String response = new String(responseData.readAllBytes(), StandardCharsets.UTF_8).trim();
        LOGGER.debug("Raw ClamAV response: '{}'", response);

        // Check for error response
        if (ERROR.isInResponse(response)) {
            return VirusScanResult.ofScanFailed("ClamAV scan error: " + response);
        }

        // Check for scan results
        if (STREAM_PREFIX.isStartOfResponse(response)) {
            return parseScanResult(response);
        }

        // Unknown response format
        LOGGER.warn("Unexpected ClamAV response format: '{}'", response);
        return VirusScanResult.ofScanFailed("Unknown response format: " + response);
    }

    /**
     * Parses a scan result response from ClamAV.
     *
     * <p>This method handles the following responses:
     *
     * <ul>
     *   <li>Clean file: Response ends with "OK"
     *   <li>Infected file: Response contains virus name followed by "FOUND"
     *   <li>Unknown format: Any other response format
     * </ul>
     *
     * @param scanResponse The raw scan response from ClamAV
     * @return A VirusScanResult indicating whether malware was found
     */
    public VirusScanResult parseScanResult(String scanResponse) {
        // Clean file
        if (OK.isEndOfResponse(scanResponse)) {
            return VirusScanResult.ofClean(scanResponse);
        }

        // Infected file
        if (FOUND.isInResponse(scanResponse)) {
            return VirusScanResult.ofInfected(scanResponse, extractVirusSignature(scanResponse));
        }

        // Unknown format
        LOGGER.warn("Unexpected stream result format: '{}'", scanResponse);
        return VirusScanResult.ofScanFailed("Unknown stream response format: " + scanResponse);
    }

    /**
     * Extracts virus name from a "FOUND" response.
     *
     * <p>The virus name is expected to be between the colon and "FOUND" in the response. For example,
     * in "stream: Win.Test.EICAR_HDB-1 FOUND", the virus name is "Win.Test.EICAR_HDB-1".
     *
     * @param response The raw ClamAV response containing a virus signature
     * @return The extracted virus name, or "Unknown" if extraction fails
     */
    public String extractVirusSignature(String response) {
        try {
            int colonIndex = response.indexOf(":");
            int foundIndex = response.indexOf(FOUND.getValue());
            if (colonIndex >= 0 && foundIndex > colonIndex) {
                return response.substring(colonIndex + 1, foundIndex).trim();
            }
        } catch (IndexOutOfBoundsException e) {
            LOGGER.warn("Failed to extract virus name from response: {}", response, e);
        }
        return "Unknown";
    }
}
