package dev.fitko.fitconnect.core.validation.virusscan;

import dev.fitko.fitconnect.api.config.virusscan.VirusScannerConfig;
import dev.fitko.fitconnect.api.config.virusscan.VirusScannerMode;
import dev.fitko.fitconnect.api.exceptions.internal.VirusScanException;
import dev.fitko.fitconnect.api.services.validation.VirusScanService;
import dev.fitko.fitconnect.core.validation.virusscan.deamon.ClamAVDeamonConfig;
import dev.fitko.fitconnect.core.validation.virusscan.process.ClamAVProcessConfig;
import java.util.Objects;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Factory for creating virus scanner implementations based on type and configuration.
 *
 * <p>This factory supports creating different types of virus scanners:
 *
 * <ul>
 *   <li>{@link VirusScannerMode#CLAMAV_DAEMON} - ClamAV daemon scanner
 *   <li>{@link VirusScannerMode#CLAMAV_PROCESS} - ClamAV process scanner
 *   <li>{@link VirusScannerMode#NO_OP} - No-op scanner for pass through scanning
 * </ul>
 */
public final class VirusScannerFactory {

    private static final Logger LOGGER = LoggerFactory.getLogger(VirusScannerFactory.class);

    private VirusScannerFactory() {
        // Utility class
    }

    /**
     * Creates a virus scanner with custom configuration based on the scanner type.
     *
     * @param scanMode the type of virus scanner to create
     * @return a configured virus scanner instance
     * @throws VirusScanException if the scanner type is not supported or creation fails
     */
    public static VirusScanService createScanner(VirusScannerMode scanMode, VirusScannerConfig virusScannerConfig)
            throws VirusScanException {
        Objects.requireNonNull(scanMode, "Scan mode must not be null");
        LOGGER.info("Creating virus scanner of type: {}", scanMode.getValue());
        try {
            switch (scanMode) {
                case CLAMAV_DAEMON:
                    return buildClamAVDeamonScanner(virusScannerConfig);
                case CLAMAV_PROCESS:
                    return buildClamAVProcessScanner(virusScannerConfig);
                case NO_OP:
                    return new NoOpVirusScanner();
                default:
                    throw new VirusScanException("Unsupported virus scanner type: " + scanMode);
            }
        } catch (Exception e) {
            LOGGER.error("Failed to create virus scanner of type: {}", scanMode, e);
            throw new VirusScanException("Failed to create virus scanner: " + e.getMessage(), e);
        }
    }

    private static ClamAVProcessScanner buildClamAVProcessScanner(VirusScannerConfig virusScannerConfig) {
        return new ClamAVProcessScanner(Optional.ofNullable(virusScannerConfig.getClamAVProcessConfig())
                .orElse(ClamAVProcessConfig.defaultConfig()));
    }

    private static ClamAVDeamonScanner buildClamAVDeamonScanner(VirusScannerConfig virusScannerConfig) {
        return new ClamAVDeamonScanner(Optional.ofNullable(virusScannerConfig.getClamAVDeamonConfig())
                .orElse(ClamAVDeamonConfig.defaultConfig()));
    }
}
