package dev.fitko.fitconnect.core.validation.virusscan;

import dev.fitko.fitconnect.api.domain.validation.VirusScanEngineInfo;
import dev.fitko.fitconnect.api.domain.validation.VirusScanResult;
import dev.fitko.fitconnect.api.exceptions.internal.VirusScanException;
import dev.fitko.fitconnect.api.services.validation.VirusScanService;
import java.io.InputStream;
import java.nio.file.Path;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * No-op virus scanner implementation that performs no actual scanning. All scan operations return a
 * clean result indicating no malware was detected.
 */
public class NoOpVirusScanner implements VirusScanService {

    private static final Logger LOGGER = LoggerFactory.getLogger(NoOpVirusScanner.class);
    private static final String NO_OP_RESULT = "No-op scanner: No malware detected";
    private static final VirusScanEngineInfo NO_OP_ENGINE_INFO = new VirusScanEngineInfo("NoOp Scanner", "1.0.0");

    @Override
    public VirusScanResult scanBytes(byte[] data) throws VirusScanException {
        LOGGER.debug("No-op virus scan for byte array of size: {}", data != null ? data.length : 0);
        return VirusScanResult.ofClean(NO_OP_RESULT);
    }

    @Override
    public VirusScanResult scanStream(InputStream inputStream) throws VirusScanException {
        LOGGER.debug("No-op virus scan for input stream");
        return VirusScanResult.ofClean(NO_OP_RESULT);
    }

    @Override
    public VirusScanResult scanFile(Path filePath) throws VirusScanException {
        LOGGER.debug("No-op virus scan for file: {}", filePath);
        return VirusScanResult.ofClean(NO_OP_RESULT);
    }

    @Override
    public boolean isAvailable() {
        LOGGER.debug("No-op virus scanner is available");
        return true;
    }

    @Override
    public VirusScanEngineInfo getEngineInfo() {
        return NO_OP_ENGINE_INFO;
    }
}
