package dev.fitko.fitconnect.core.validation.virusscan;

import static dev.fitko.fitconnect.core.utils.Preconditions.checkArgument;

import dev.fitko.fitconnect.api.domain.validation.VirusScanEngineInfo;
import dev.fitko.fitconnect.api.domain.validation.VirusScanResult;
import dev.fitko.fitconnect.api.exceptions.internal.VirusScanException;
import dev.fitko.fitconnect.api.services.validation.VirusScanService;
import dev.fitko.fitconnect.core.io.ProcessRunner;
import dev.fitko.fitconnect.core.utils.StopWatch;
import dev.fitko.fitconnect.core.validation.virusscan.process.ClamAVProcessConfig;
import dev.fitko.fitconnect.core.validation.virusscan.process.ClamAVProcessResultParser;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * ClamAV process virus scanner implementation.
 *
 * <p>Executes ClamAV clamscan process to scan files. This implementation is useful when ClamAV
 * daemon is not available or not preferred.
 */
public class ClamAVProcessScanner implements VirusScanService {

    private static final Logger LOGGER = LoggerFactory.getLogger(ClamAVProcessScanner.class);
    private final ClamAVProcessConfig config;
    private final ClamAVProcessResultParser resultParser;

    public ClamAVProcessScanner(ClamAVProcessConfig config) {
        this.config = config != null ? config : ClamAVProcessConfig.defaultConfig();
        this.resultParser = new ClamAVProcessResultParser();
        LOGGER.info("Initialized ClamAV process scanner with executable: {}", this.config.getExecutablePath());
    }

    @Override
    public VirusScanResult scanBytes(byte[] data) throws VirusScanException {
        checkArgument(data == null, () -> new VirusScanException("Data bytes must not be null"));
        LOGGER.debug("Scanning byte array of size: {}", data.length);

        try {
            Path tempFile = createTemporaryFile();
            try {
                Files.write(tempFile, data, StandardOpenOption.CREATE);
                return scanFile(tempFile);
            } finally {
                cleanupTemporaryFile(tempFile);
            }
        } catch (IOException e) {
            throw new VirusScanException("Failed to create temporary file for scanning", e);
        }
    }

    @Override
    public VirusScanResult scanStream(InputStream inputStream) throws VirusScanException {
        checkArgument(inputStream == null, () -> new VirusScanException("Input stream must not be null"));
        LOGGER.debug("Scanning input stream");

        try {
            Path tempFile = createTemporaryFile();
            try {
                Files.copy(inputStream, tempFile, StandardCopyOption.REPLACE_EXISTING);
                return scanFile(tempFile);
            } finally {
                cleanupTemporaryFile(tempFile);
            }
        } catch (IOException e) {
            throw new VirusScanException("Failed to create temporary file for scanning", e);
        }
    }

    @Override
    public VirusScanResult scanFile(Path filePath) throws VirusScanException {
        checkArgument(filePath == null, () -> new VirusScanException("File path must not be null"));

        if (!Files.exists(filePath)) {
            throw new VirusScanException("File does not exist: " + filePath);
        }

        LOGGER.debug("Scanning file: {}", filePath);

        // Build command
        List<String> command = buildScanCommand(filePath.toString());

        // Execute scan
        final long start = StopWatch.start();
        ProcessRunner.Result result = ProcessRunner.runCommand(command);
        LOGGER.debug("Scanning for viruses took {}", StopWatch.stop(start));

        // Parse result
        return resultParser.parseResult(result);
    }

    @Override
    public boolean isAvailable() {
        try {
            ProcessRunner.Result result = ProcessRunner.runCommand(List.of(config.getExecutablePath(), "--version"));
            return result.exitCode == 0 && result.output != null && result.output.contains("ClamAV");
        } catch (Exception e) {
            LOGGER.debug("ClamAV availability check failed: {}", e.getMessage());
            return false;
        }
    }

    @Override
    public VirusScanEngineInfo getEngineInfo() {
        try {
            ProcessRunner.Result result = ProcessRunner.runCommand(List.of(config.getExecutablePath(), "--version"));
            if (result.exitCode == 0 && result.output != null && result.output.contains("ClamAV")) {
                String version = Arrays.stream(result.output.split("\\r?\\n"))
                        .map(String::trim)
                        .filter(line -> line.startsWith("ClamAV"))
                        .findFirst()
                        .orElse(null);
                return new VirusScanEngineInfo("ClamAV", version != null ? version : result.output.trim());
            }
        } catch (Exception e) {
            LOGGER.error("Failed to get ClamAV version info", e);
        }
        return null;
    }

    /**
     * Builds the command list for scanning a file with clamscan.
     *
     * @param filePath the path to the file to scan
     * @return the command list for ProcessRunner
     */
    private List<String> buildScanCommand(String filePath) {
        List<String> command = List.of(config.getExecutablePath(), filePath);
        LOGGER.debug("Built ClamAV command: {}", String.join(" ", command));
        return command;
    }

    /**
     * Creates a temporary file for scanning.
     *
     * @return the path to the created temporary file
     * @throws IOException if file creation fails
     */
    private Path createTemporaryFile() throws IOException {

        Path tempDir = Path.of(config.getTempDirectory());

        if (!Files.exists(tempDir)) {
            Files.createDirectories(tempDir);
            LOGGER.debug("Created custom temporary directory: {}", tempDir);
        }

        if (!Files.isWritable(tempDir)) {
            throw new IOException("Custom temporary directory is not writable: " + tempDir);
        }

        return Files.createTempFile("clamav_scan_", ".tmp");
    }

    private void cleanupTemporaryFile(Path tempFile) {
        try {
            Files.deleteIfExists(tempFile);
        } catch (IOException e) {
            LOGGER.warn("Failed to delete temporary file: {}", tempFile, e);
        }
    }
}
