package dev.fitko.fitconnect.core.validation.virusscan;

import dev.fitko.fitconnect.api.domain.validation.VirusScanEngineInfo;
import dev.fitko.fitconnect.api.domain.validation.VirusScanResult;
import dev.fitko.fitconnect.api.exceptions.internal.VirusScanException;
import dev.fitko.fitconnect.api.services.validation.VirusScanService;
import dev.fitko.fitconnect.core.utils.Preconditions;
import dev.fitko.fitconnect.core.utils.StopWatch;
import dev.fitko.fitconnect.core.validation.virusscan.deamon.ClamAVDeamonConfig;
import dev.fitko.fitconnect.core.validation.virusscan.deamon.ClamAVStreamCommand;
import dev.fitko.fitconnect.core.validation.virusscan.deamon.response.ClamAVDeamonResultParser;
import dev.fitko.fitconnect.core.validation.virusscan.deamon.socket.ClamAVSocketFactory;
import dev.fitko.fitconnect.core.validation.virusscan.deamon.socket.DefaultClamAVSocketFactory;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.Socket;
import java.nio.file.Files;
import java.nio.file.Path;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * ClamAV deamon virus scanner implementation.
 *
 * <p>Accesses a running ClamAV deamon via TCP Sockets to send commands and receive scan results.
 *
 * @see <a href="https://manpages.debian.org/testing/clamav-daemon/clamd.8.en.html">Clam Deamon
 *     Documentation</a>
 * @see <a href="https://docs.clamav.net">ClamAV Documentation</a>}
 */
public class ClamAVDeamonScanner implements VirusScanService {

    private static final Logger LOGGER = LoggerFactory.getLogger(ClamAVDeamonScanner.class);
    private final ClamAVDeamonConfig config;
    private final ClamAVDeamonResultParser resultHandler;
    private final ClamAVSocketFactory socketFactory;

    /** Default Constructor with default configuration. */
    public ClamAVDeamonScanner() {
        this(ClamAVDeamonConfig.defaultConfig(), new ClamAVDeamonResultParser(), new DefaultClamAVSocketFactory());
    }

    /**
     * Constructor with custom configuration.
     *
     * @param config the ClamAV configuration
     */
    public ClamAVDeamonScanner(final ClamAVDeamonConfig config) {
        this.config = config != null ? config : ClamAVDeamonConfig.defaultConfig();
        this.resultHandler = new ClamAVDeamonResultParser();
        this.socketFactory = new DefaultClamAVSocketFactory();
        LOGGER.info("Setup new ClamAV scanner at {}:{}", this.config.getHost(), this.config.getPort());
    }

    /**
     * Constructor with custom configuration, result-handler and socketFactory.
     *
     * @param config the ClamAV configuration
     * @param resultHandler a handler to evaluate the result from ClamAV
     * @param socketFactory a factory that creates and connects to a socket for ClamAV communication
     */
    public ClamAVDeamonScanner(
            final ClamAVDeamonConfig config,
            ClamAVDeamonResultParser resultHandler,
            ClamAVSocketFactory socketFactory) {
        this.config = config != null ? config : ClamAVDeamonConfig.defaultConfig();
        this.resultHandler = resultHandler;
        this.socketFactory = socketFactory;
        LOGGER.info("Setup new ClamAV scanner at {}:{}", this.config.getHost(), this.config.getPort());
    }

    @Override
    public VirusScanResult scanBytes(byte[] data) throws VirusScanException {
        Preconditions.checkArgument(data == null, () -> new VirusScanException("data bytes must not be null"));
        return scanStream(new ByteArrayInputStream(data));
    }

    @Override
    public VirusScanResult scanFile(Path filePath) throws VirusScanException {
        Preconditions.checkArgument(filePath == null, () -> new VirusScanException("filePath must not be null"));
        try {
            return scanStream(Files.newInputStream(filePath));
        } catch (IOException e) {
            LOGGER.error("Reading file from path {} failed", filePath);
            throw new VirusScanException(e.getMessage(), e);
        }
    }

    @Override
    public VirusScanResult scanStream(InputStream inputStream) throws VirusScanException {
        Preconditions.checkArgument(inputStream == null, () -> new VirusScanException("inputStream must not be null"));

        try (Socket socket = socketFactory.createSocket(config);
                OutputStream socketOutStream = new BufferedOutputStream(socket.getOutputStream());
                InputStream socketInputStream = socket.getInputStream()) {

            final long start = StopWatch.start();
            streamDataToClamAV(inputStream, socketOutStream);
            LOGGER.debug("Scanning for viruses took {}", StopWatch.stop(start));

            return resultHandler.parseClamAVResponse(socketInputStream);

        } catch (Exception e) {
            LOGGER.error("Scan failed with ClamAV at {}:{}, {}", config.getHost(), config.getPort(), e.getMessage());
            throw new VirusScanException("Virus scan failed: " + e.getMessage(), e);
        }
    }

    @Override
    public boolean isAvailable() {
        try {
            final String response = sendCommandToClamAV(ClamAVStreamCommand.PING);
            return response != null && response.trim().equalsIgnoreCase(ClamAVStreamCommand.PONG.getAsString());
        } catch (IOException e) {
            LOGGER.error("Network error connecting to ClamAV deamon {}:{}", config.getHost(), config.getPort(), e);
            return false;
        }
    }

    @Override
    public VirusScanEngineInfo getEngineInfo() {
        try {
            return new VirusScanEngineInfo("ClamAV", sendCommandToClamAV(ClamAVStreamCommand.VERSION));
        } catch (Exception e) {
            LOGGER.error("Failed to get ClamAV version info", e);
        }
        return null;
    }

    /**
     * Sends a text command to ClamAV and returns the response.
     *
     * @param cmd The ClamAV command to send
     * @return Single-line response from daemon
     * @throws IOException if connection fails
     */
    private String sendCommandToClamAV(final ClamAVStreamCommand cmd) throws IOException {
        try (Socket socket = socketFactory.createSocket(config);
                OutputStream out = socket.getOutputStream();
                BufferedReader reader = new BufferedReader(new InputStreamReader(socket.getInputStream()))) {
            // Send command
            out.write(cmd.getAsBytes());
            out.flush();
            // Get response
            return reader.readLine();
        }
    }

    /**
     * Implements the ClamAV INSTREAM protocol to scan data without saving to disk.
     *
     * <p>This method streams data to ClamAV in the following format:
     *
     * <ul>
     *   <li>INSTREAM command: <code>zINSTREAM\0</code>
     *   <li>Data chunks: <code>[4-byte size][chunk data][4-byte size][chunk data]...</code>
     *   <li>End marker: <code>[0,0,0,0]</code>
     * </ul>
     *
     * @param inputStream The data stream to scan
     * @param outStream The output stream connected to ClamAV daemon
     * @throws IOException if communication with ClamAV fails
     * @see <a href="https://manpages.debian.org/testing/clamav-daemon/clamd.8.en.html#INSTREAM">clamd
     *     INSTREAM man page</a>
     */
    private void streamDataToClamAV(InputStream inputStream, OutputStream outStream) throws IOException {

        // #1 Send INSTREAM command to initiate streaming protocol
        outStream.write(ClamAVStreamCommand.IN_STREAM.getAsBytes());
        outStream.flush();

        byte[] readingBuffer = new byte[config.getReadBufferSize()];
        final DataOutputStream dataOutputStream = new DataOutputStream(outStream);

        // #2 Stream data in chunks with size headers
        int bytesRead;
        while ((bytesRead = inputStream.read(readingBuffer)) != -1) {
            // Send chunk size header (4 bytes)
            dataOutputStream.writeInt(bytesRead);
            // Send actual data bytes
            dataOutputStream.write(readingBuffer, 0, bytesRead);
        }

        // #3 Send END_STREAM command to signal completion
        outStream.write(ClamAVStreamCommand.END_STREAM.getAsBytes());
        outStream.flush();
    }
}
