package dev.fitko.fitconnect.core.validation;

import static com.networknt.schema.SpecificationVersion.*;
import static dev.fitko.fitconnect.core.crypto.constants.CryptoConstants.HASH_OF_ZERO_BYTES;
import static java.util.Objects.isNull;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonFactoryBuilder;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.StreamReadConstraints;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.networknt.schema.Error;
import com.networknt.schema.SchemaRegistry;
import com.networknt.schema.SchemaRegistryConfig;
import com.nimbusds.jose.jwk.KeyOperation;
import com.nimbusds.jose.jwk.RSAKey;
import dev.fitko.fitconnect.api.config.ApplicationConfig;
import dev.fitko.fitconnect.api.domain.model.attachment.Fragment;
import dev.fitko.fitconnect.api.domain.model.destination.DestinationReplyChannels;
import dev.fitko.fitconnect.api.domain.model.destination.PublicDestination;
import dev.fitko.fitconnect.api.domain.model.destination.Service;
import dev.fitko.fitconnect.api.domain.model.event.EventClaimFields;
import dev.fitko.fitconnect.api.domain.model.event.authtags.AuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.event.problems.Problem;
import dev.fitko.fitconnect.api.domain.model.event.problems.attachment.AttachmentHashMismatch;
import dev.fitko.fitconnect.api.domain.model.event.problems.attachment.IncorrectAttachmentAuthenticationTag;
import dev.fitko.fitconnect.api.domain.model.event.problems.data.*;
import dev.fitko.fitconnect.api.domain.model.event.problems.metadata.*;
import dev.fitko.fitconnect.api.domain.model.metadata.Metadata;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.ApiAttachment;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.AttachmentForValidation;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.Purpose;
import dev.fitko.fitconnect.api.domain.model.metadata.data.Data;
import dev.fitko.fitconnect.api.domain.model.metadata.data.MimeType;
import dev.fitko.fitconnect.api.domain.model.metadata.data.SubmissionSchema;
import dev.fitko.fitconnect.api.domain.model.reply.Reply;
import dev.fitko.fitconnect.api.domain.model.submission.PublicService;
import dev.fitko.fitconnect.api.domain.model.submission.Submission;
import dev.fitko.fitconnect.api.domain.validation.ValidationResult;
import dev.fitko.fitconnect.api.exceptions.internal.DataIntegrityException;
import dev.fitko.fitconnect.api.exceptions.internal.SchemaNotFoundException;
import dev.fitko.fitconnect.api.exceptions.internal.ValidationException;
import dev.fitko.fitconnect.api.services.crypto.MessageDigestService;
import dev.fitko.fitconnect.api.services.schema.SchemaProvider;
import dev.fitko.fitconnect.api.services.validation.ValidationService;
import dev.fitko.fitconnect.core.validation.xml.XmlSchemaValidator;
import dev.fitko.fitconnect.jwkvalidator.JWKValidator;
import dev.fitko.fitconnect.jwkvalidator.exceptions.JWKValidationException;
import dev.fitko.fitconnect.jwkvalidator.exceptions.LogLevel;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.file.Files;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.*;
import java.util.stream.Collectors;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParserFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

public class DefaultValidationService implements ValidationService {

    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultValidationService.class);
    private static final ObjectMapper MAPPER = createObjectMapper();
    private static final SchemaRegistryConfig VALIDATORS_CONFIG = SchemaRegistryConfig.builder()
            .formatAssertionsEnabled(true)
            .locale(Locale.US)
            .build();
    private static final SchemaRegistry SCHEMA_REGISTRY_DRAFT_2020 = SchemaRegistry.withDefaultDialect(
            DRAFT_2020_12, builder -> builder.schemaRegistryConfig(VALIDATORS_CONFIG));
    private static final SchemaRegistry SCHEMA_REGISTRY_DRAFT_2017 =
            SchemaRegistry.withDefaultDialect(DRAFT_7, builder -> builder.schemaRegistryConfig(VALIDATORS_CONFIG));

    private final MessageDigestService messageDigestService;
    private final SchemaProvider schemaProvider;
    private final ApplicationConfig config;
    private final JWKValidator jwkValidator;
    private final XmlSchemaValidator xmlSchemaValidator;

    public DefaultValidationService(
            final ApplicationConfig config,
            final MessageDigestService messageDigestService,
            final SchemaProvider schemaProvider,
            final XmlSchemaValidator xmlSchemaValidator,
            final List<String> trustedRootCertificates) {
        this.config = config;
        this.messageDigestService = messageDigestService;
        this.schemaProvider = schemaProvider;
        this.xmlSchemaValidator = xmlSchemaValidator;
        this.jwkValidator = createValidator(config, trustedRootCertificates);
    }

    @Override
    public ValidationResult validatePublicKey(final RSAKey publicKey, final KeyOperation keyOperation) {
        return validatePublicKey(publicKey, null, keyOperation);
    }

    @Override
    public ValidationResult validatePublicKey(RSAKey publicKey, Date validationDate, KeyOperation keyOperation) {
        try {
            return validateKey(publicKey, validationDate, keyOperation);
        } catch (final Exception e) {
            return ValidationResult.error(e);
        }
    }

    @Override
    public ValidationResult validateSetEventSchema(final String setEventPayload) {
        try {
            final JsonNode inputNode = MAPPER.readTree(setEventPayload);
            final URI schemaUri =
                    URI.create(inputNode.get(EventClaimFields.CLAIM_SCHEMA).asText());
            if (schemaProvider.isAllowedSetSchema(schemaUri)) {
                return validate2020JsonSchema(
                        schemaProvider.loadSetSchema(config.getSetSchemaWriteVersion()), inputNode);
            } else {
                return ValidationResult.error(
                        new SchemaNotFoundException("SET payload schema not supported: " + schemaUri));
            }
        } catch (final JsonProcessingException | IllegalArgumentException e) {
            return ValidationResult.error(e);
        }
    }

    @Override
    public ValidationResult validateMetadataSchema(final Metadata metadata) {
        // some schema versions don't have a mandatory $schema object
        if (metadata.getSchema() == null) {
            return ValidationResult.ok();
        }

        final ValidationResult schemaAllowance = validateSchemaAllowance(metadata.getSchema());
        if (schemaAllowance.hasProblems()) {
            return schemaAllowance;
        }

        return loadSchemaAndValidate(metadata);
    }

    @Override
    public ValidationResult validateDestinationSchema(final Map<String, Object> destinationPayload) {
        try {
            final String destinationPayloadJson = MAPPER.writeValueAsString(destinationPayload);
            final JsonNode inputNode = MAPPER.readTree(destinationPayloadJson);
            final String schema = schemaProvider.loadDestinationSchema(config.getDestinationSchemaUri());
            return returnValidationResult(
                    SCHEMA_REGISTRY_DRAFT_2017.getSchema(schema).validate(inputNode));
        } catch (final JsonProcessingException e) {
            return ValidationResult.error(e);
        }
    }

    @Override
    public ValidationResult validateHashIntegrity(final String originalHexHash, final byte[] data) {
        try {
            final byte[] originalHash = messageDigestService.fromHexString(originalHexHash);
            final boolean hashesAreNotEqual = !messageDigestService.verify(originalHash, data);
            if (hashesAreNotEqual) {
                return ValidationResult.error(new DataIntegrityException("Metadata contains invalid hash value"));
            }
            return ValidationResult.ok();
        } catch (final IllegalArgumentException | NullPointerException e) {
            return ValidationResult.error(e);
        }
    }

    @Override
    public ValidationResult validateHashIntegrity(final String originalHexHash, final InputStream inputStream) {
        try {
            final byte[] originalHash = messageDigestService.fromHexString(originalHexHash);
            final boolean hashesAreNotEqual = !messageDigestService.verify(originalHash, inputStream);
            if (hashesAreNotEqual) {
                return ValidationResult.error(new DataIntegrityException("Metadata contains invalid hash value"));
            }
            return ValidationResult.ok();
        } catch (final IllegalArgumentException | NullPointerException e) {
            return ValidationResult.error(e);
        }
    }

    @Override
    public ValidationResult validateJsonSubmissionDataSchema(final byte[] json, final URI schemaUri) {

        if (config.isSkipSubmissionDataValidation()) {
            LOGGER.warn(
                    "Submission data validation is deactivated. This should be done only on secure test environments.");
            return ValidationResult.ok();
        }

        final String schema = schemaProvider.loadSubmissionDataSchema(schemaUri);
        try {
            return returnValidationResult(
                    SCHEMA_REGISTRY_DRAFT_2020.getSchema(schema).validate(MAPPER.readTree(json)));
        } catch (final IOException e) {
            return ValidationResult.error(e);
        }
    }

    @Override
    public ValidationResult validateXmlSubmissionDataSchema(final byte[] xml, final URI schemaUri) {
        if (config.isSkipSubmissionDataValidation()) {
            LOGGER.warn(
                    "Submission data validation is deactivated. This should be done only on secure test environments.");
            return ValidationResult.ok();
        }
        return xmlSchemaValidator.validate(xml, schemaUri);
    }

    @Override
    public ValidationResult validateJsonFormat(final byte[] json) {
        try {
            MAPPER.readTree(json);
        } catch (final IOException e) {
            return ValidationResult.error(e);
        }
        return ValidationResult.ok();
    }

    @Override
    public ValidationResult validateXmlFormat(final byte[] xml) {
        try {
            final XMLReader xmlReader = getXmlReader();
            xmlReader.parse(new InputSource(new ByteArrayInputStream(xml)));
        } catch (final ParserConfigurationException | IOException | SAXException e) {
            return ValidationResult.error(e);
        }
        return ValidationResult.ok();
    }

    @Override
    public ValidationResult validateCallback(
            final String hmac, final Long timestampInSec, final String httpBody, final String callbackSecret) {

        final ZonedDateTime providedTimeInSeconds =
                ZonedDateTime.ofInstant(Instant.ofEpochSecond(timestampInSec), ZoneId.systemDefault());
        final ZonedDateTime currentTimeFiveMinutesAgo = ZonedDateTime.now().minusMinutes(5);

        if (providedTimeInSeconds.isBefore(currentTimeFiveMinutesAgo)) {
            return ValidationResult.error(new ValidationException("Timestamp provided by callback is expired."));
        }

        final String expectedHmac = messageDigestService.calculateHMAC(timestampInSec + "." + httpBody, callbackSecret);

        if (!hmac.equals(expectedHmac)) {
            return ValidationResult.error(
                    new ValidationException("HMAC provided by callback does not match the expected result."));
        }

        return ValidationResult.ok();
    }

    @Override
    public ValidationResult validateData(
            final byte[] decryptedData,
            final String encryptedData,
            final Metadata metadata,
            final String eventAuthenticationTag) {

        // https://docs.fitko.de/fit-connect/docs/receiving/verification/#authentication-tag-pr%C3%BCfen-1
        final var dataAuthTag = AuthenticationTags.getAuthTagFromJWT(encryptedData);
        if (!eventAuthenticationTag.equals(dataAuthTag)) {
            return ValidationResult.problem(new IncorrectDataAuthenticationTag());
        }

        final Data data = metadata.getContentStructure().getData();

        if (validateIfDataIsNotTransferredAsAttachment(metadata, data)) {
            // https://docs.fitko.de/fit-connect/docs/receiving/verification/#submission-data-hash
            final String hashFromSender = data.getHash().getContent();
            final ValidationResult hashValidation = validateHashIntegrity(hashFromSender, decryptedData);
            if (hashValidation.hasError()) {
                return ValidationResult.problem(new DataHashMismatch());
            }
        }

        // https://docs.fitko.de/fit-connect/docs/receiving/verification/#syntax-validierung-1
        if (data.getSubmissionSchema().getMimeType().equals(MimeType.APPLICATION_JSON)) {

            final ValidationResult jsonValidation = validateJsonFormat(decryptedData);
            if (jsonValidation.hasError()) {
                return ValidationResult.problem(new DataJsonSyntaxViolation());
            }

            final ValidationResult dataSchemaValidation = validateJsonSubmissionDataSchema(
                    decryptedData, data.getSubmissionSchema().getSchemaUri());
            if (dataSchemaValidation.hasError()) {
                return ValidationResult.problem(new DataSchemaViolation());
            }
        }

        if (data.getSubmissionSchema().getMimeType().equals(MimeType.APPLICATION_XML)) {

            final ValidationResult xmlValidation = validateXmlFormat(decryptedData);
            if (xmlValidation.hasError()) {
                return ValidationResult.problem(new DataXmlSyntaxViolation());
            }

            final ValidationResult dataSchemaValidation = validateXmlSubmissionDataSchema(
                    decryptedData, data.getSubmissionSchema().getSchemaUri());
            if (dataSchemaValidation.hasError()) {
                return ValidationResult.problem(new DataSchemaViolation());
            }
        }

        return ValidationResult.ok();
    }

    @Override
    public ValidationResult validateAttachments(
            final List<AttachmentForValidation> attachmentsForValidation, final AuthenticationTags authenticationTags) {

        final Map<UUID, String> eventAuthTags = authenticationTags.getAttachments();
        final List<Problem> validationProblems = new ArrayList<>();

        for (AttachmentForValidation attachment : attachmentsForValidation) {
            if (attachment.hasFragmentedPayload()) {
                final ValidationResult validationResult =
                        validateFragmentedAttachment(attachment, eventAuthTags, validationProblems);
                if (validationResult.hasError()) {
                    return validationResult;
                }
            } else {
                validateAttachment(attachment, eventAuthTags, validationProblems);
            }
        }

        return validationProblems.isEmpty() ? ValidationResult.ok() : ValidationResult.problems(validationProblems);
    }

    private void validateAttachment(
            AttachmentForValidation attachment, Map<UUID, String> eventAuthTags, List<Problem> validationProblems) {
        final UUID attachmentId = attachment.getAttachmentId();
        // https://docs.fitko.de/fit-connect/docs/receiving/verification/#authentication-tag-pr%C3%BCfen-2
        final String authTagFromEvent = eventAuthTags.get(attachmentId);
        if (!authTagFromEvent.equals(attachment.getAuthTag())) {
            validationProblems.add(new IncorrectAttachmentAuthenticationTag(attachmentId));
        }
        // https://docs.fitko.de/fit-connect/docs/receiving/verification/#attachment-hash
        final ValidationResult validationResult =
                validateHashIntegrity(attachment.getHash(), attachment.getDecryptedData());
        if (validationResult.hasError()) {
            validationProblems.add(new AttachmentHashMismatch(attachment.getAttachmentId()));
        }
    }

    private ValidationResult validateFragmentedAttachment(
            AttachmentForValidation attachment, Map<UUID, String> eventAuthTags, List<Problem> validationProblems) {
        for (final Fragment fragment : attachment.getFragments()) {
            // https://docs.fitko.de/fit-connect/docs/receiving/verification/#authentication-tag-pr%C3%BCfen-2
            final String authTagFromEvent = eventAuthTags.get(fragment.getFragmentId());
            if (!authTagFromEvent.equals(fragment.getAuthTag())) {
                validationProblems.add(new IncorrectAttachmentAuthenticationTag(fragment.getFragmentId()));
            }
        }
        // https://docs.fitko.de/fit-connect/docs/receiving/verification/#attachment-hash
        try (InputStream is = Files.newInputStream(attachment.getDataFile().toPath())) {
            final ValidationResult validationResult = validateHashIntegrity(attachment.getHash(), is);
            if (validationResult.hasError()) {
                validationProblems.add(new AttachmentHashMismatch(attachment.getAttachmentId()));
            }
        } catch (final IOException e) {
            return ValidationResult.error(e);
        }
        return ValidationResult.ok();
    }

    @Override
    public ValidationResult validateSubmissionMetadata(
            final Metadata metadata,
            final Submission submission,
            final PublicDestination destination,
            final AuthenticationTags eventAuthenticationTags) {
        return validateMetadata(
                metadata,
                submission.getEncryptedMetadata(),
                destination,
                submission.getPublicService(),
                submission.getAttachments(),
                eventAuthenticationTags);
    }

    @Override
    public ValidationResult validateReplyMetadata(
            final Metadata metadata,
            final Reply reply,
            final PublicDestination destination,
            final AuthenticationTags eventAuthenticationTags) {
        return validateMetadata(
                metadata,
                reply.getEncryptedMetadata(),
                destination,
                null,
                reply.getAttachments(),
                eventAuthenticationTags);
    }

    private ValidationResult validateMetadata(
            final Metadata metadata,
            final String encryptedMetadata,
            final PublicDestination destination,
            final PublicService serviceType,
            final List<UUID> attachments,
            final AuthenticationTags eventAuthenticationTags) {

        // https://docs.fitko.de/fit-connect/docs/receiving/verification/#authentication-tag-pr%C3%BCfen
        final var metadataAuthTag = AuthenticationTags.getAuthTagFromJWT(encryptedMetadata);
        if (!eventAuthenticationTags.getMetadata().equals(metadataAuthTag)) {
            return ValidationResult.problem(new IncorrectMetadataAuthenticationTag());
        }

        // https://docs.fitko.de/fit-connect/docs/receiving/verification/#syntax-validierung
        if (invalidJsonSyntax(metadata)) {
            return ValidationResult.problem(new MetadataJsonSyntaxViolation());
        }

        // https://docs.fitko.de/fit-connect/docs/receiving/verification/#fachdatensatz
        if (metadata.getContentStructure().getData() == null) {
            return ValidationResult.problem(new MissingData());
        }

        final ValidationResult validationResult = validateMetadataSchema(metadata);
        if (validationResult.hasProblems()) {
            return validationResult;
        }

        // https://docs.fitko.de/fit-connect/docs/receiving/verification/#fachdatenschema
        if (destination.getPublicServices().isEmpty()) {
            return ValidationResult.problem(new UnsupportedDataSchema());
        }
        final SubmissionSchema submissionSchema =
                metadata.getContentStructure().getData().getSubmissionSchema();
        final boolean matchingSchemas =
                matchingDestinationAndSubmissionSchema(destination, submissionSchema.getSchemaUri());
        if (!matchingSchemas) {
            return ValidationResult.problem(new UnsupportedDataSchema());
        }

        // https://docs.fitko.de/fit-connect/docs/receiving/verification/#verwaltungsleistung-abgleichen-ii
        if (serviceType != null) {
            final var destinationService = destination.getPublicServices().stream()
                    .map(Service::getIdentifier)
                    .filter(identifier -> identifier.equals(serviceType.getIdentifier()))
                    .findFirst();

            if (destinationService.isEmpty()) {
                return ValidationResult.problem(new UnsupportedService());
            }
        }

        // https://docs.fitko.de/fit-connect/docs/receiving/verification/#liste-der-anlagen-abgleichen
        final ValidationResult attachmentValidation =
                checkAttachmentCount(attachments, metadata, eventAuthenticationTags);
        if (attachmentValidation.hasProblems()) {
            return attachmentValidation;
        }

        // https://docs.fitko.de/fit-connect/docs/receiving/verification/#r%C3%BCckkanal
        if (metadata.getReplyChannel() != null && serviceType != null) {
            final Optional<DestinationReplyChannels> matchingServiceReplyChannel =
                    destination.getPublicServices().stream()
                            .filter(service -> service.getIdentifier().equals(serviceType.getIdentifier()))
                            .filter(service -> service.getSubmissionSchemas().stream()
                                    .anyMatch(schema -> schema.equals(submissionSchema)))
                            .map(Service::getReplyChannels)
                            .filter(Objects::nonNull)
                            .filter(destinationReplyChannel ->
                                    destinationReplyChannel.allowsSubmissionReplyChannel(metadata.getReplyChannel()))
                            .findFirst();

            if (matchingServiceReplyChannel.isEmpty()) {
                return ValidationResult.problem(new UnsupportedReplyChannel());
            }
        }

        return ValidationResult.ok();
    }

    private boolean validateIfDataIsNotTransferredAsAttachment(Metadata metadata, Data data) {
        final boolean noAttachmentWithDataPurpose = metadata.getContentStructure().getAttachments().stream()
                .map(ApiAttachment::getPurpose)
                .noneMatch(purpose -> purpose.equals(Purpose.DATA));
        return noAttachmentWithDataPurpose && !data.getHash().getContent().equals(HASH_OF_ZERO_BYTES);
    }

    private static boolean matchingDestinationAndSubmissionSchema(
            final PublicDestination destination, final URI submissionDataSchemaUri) {
        return destination.getPublicServices().stream()
                .flatMap(service -> service.getSubmissionSchemas().stream())
                .map(SubmissionSchema::getSchemaUri)
                .anyMatch(submissionDataSchemaUri::equals);
    }

    private ValidationResult validate2020JsonSchema(final String schema, final JsonNode inputNode) {
        return returnValidationResult(
                SCHEMA_REGISTRY_DRAFT_2020.getSchema(schema).validate(inputNode));
    }

    private ValidationResult returnValidationResult(final List<Error> errors) {
        if (errors.isEmpty()) {
            return ValidationResult.ok();
        }
        return ValidationResult.withErrorAndProblem(
                new ValidationException(errorsToSingleString(errors)), new MetadataSchemaViolation());
    }

    private ValidationResult validateKey(final RSAKey publicKey, Date validationDate, final KeyOperation purpose)
            throws JWKValidationException {
        if (config.isAllowInsecurePublicKey()) {
            return validateWithoutCertChain(publicKey, purpose);
        } else {
            return validateWithCertChain(publicKey, validationDate, purpose);
        }
    }

    private ValidationResult validateWithoutCertChain(final RSAKey publicKey, final KeyOperation purpose)
            throws JWKValidationException {
        LOGGER.debug("Validating public key {} without XC5 certificate chain", publicKey.getKeyID());
        jwkValidator.validate(publicKey, purpose);
        return ValidationResult.ok();
    }

    private ValidationResult validateWithCertChain(
            final RSAKey publicKey, Date validationDate, final KeyOperation purpose) throws JWKValidationException {
        LOGGER.info("Validating public key with x5c certificate chain checks");
        if (publicKey.getParsedX509CertChain() == null) {
            throw new IllegalStateException(
                    "Public key with id '" + publicKey.getKeyID() + "' does not contain an x5c certificate chain");
        }
        if (validationDate == null) {
            jwkValidator.validate(publicKey, purpose);
        } else {
            jwkValidator.validate(publicKey, validationDate, purpose);
        }
        return ValidationResult.ok();
    }

    private String errorsToSingleString(final List<Error> errors) {
        return errors.stream()
                .map(e -> e.getInstanceLocation().toString() + ": " + e.getMessage())
                .collect(Collectors.joining("\n"));
    }

    private static XMLReader getXmlReader() throws ParserConfigurationException, SAXException {
        final SAXParserFactory saxParserFactory = SAXParserFactory.newInstance();
        saxParserFactory.setFeature("http://apache.org/xml/features/disallow-doctype-decl", true);
        return saxParserFactory.newSAXParser().getXMLReader();
    }

    private boolean invalidJsonSyntax(final Metadata metadata) {
        try {
            return metadata == null
                    || validateJsonFormat(MAPPER.writeValueAsBytes(metadata)).hasError();
        } catch (final JsonProcessingException | NullPointerException e) {
            return true;
        }
    }

    private ValidationResult checkAttachmentCount(
            final List<UUID> attachmentIds, final Metadata metadata, final AuthenticationTags authenticationTags) {

        final List<ApiAttachment> metadataAttachments =
                metadata.getContentStructure().getAttachments();
        final Map<UUID, String> attachmentAuthTags = authenticationTags.getAttachments();

        if ((isNull(attachmentIds) || attachmentIds.isEmpty())
                && (isNull(attachmentAuthTags) || attachmentAuthTags.isEmpty())) {
            return ValidationResult.ok();
        }

        final List<UUID> actualIdsIncludingFragments = new ArrayList<>();
        for (final ApiAttachment metadataAttachment : metadataAttachments) {
            final List<UUID> fragments = metadataAttachment.getFragments();
            if (metadataAttachment.hasFragments()) {
                actualIdsIncludingFragments.addAll(fragments);
            } else {
                actualIdsIncludingFragments.add(metadataAttachment.getAttachmentId());
            }
        }

        if (actualIdsIncludingFragments.size() != attachmentIds.size()
                || attachmentAuthTags.size() != attachmentIds.size()
                || !attachmentAuthTags.keySet().containsAll(attachmentIds)
                || !attachmentIds.stream()
                        .allMatch(id -> actualIdsIncludingFragments.stream().anyMatch(actualId -> actualId.equals(id)))
                || !attachmentIds.stream().allMatch(attachmentAuthTags::containsKey)) {
            return ValidationResult.problem(new AttachmentsMismatch());
        }

        return ValidationResult.ok();
    }

    private JWKValidator createValidator(final ApplicationConfig config, final List<String> trustedRootCertificates) {
        if (config.isAllowInsecurePublicKey()) {
            return JWKValidator.withoutX5CValidation()
                    .withErrorLogLevel(LogLevel.WARN)
                    .build();
        }
        return JWKValidator.withRecommendedDefaults()
                .withProxy(config.getHttpConfig().getProxyConfig().getHttpProxy())
                .withRootCertificatesAsPEM(trustedRootCertificates)
                .build();
    }

    private static ObjectMapper createObjectMapper() {
        final JsonFactory jsonFactory = new JsonFactoryBuilder()
                .streamReadConstraints(StreamReadConstraints.builder()
                        .maxStringLength(Integer.MAX_VALUE)
                        .build())
                .build();
        return new ObjectMapper(jsonFactory)
                .setDateFormat(new SimpleDateFormat("yyyy-MM-dd"))
                .setSerializationInclusion(JsonInclude.Include.NON_NULL);
    }

    private ValidationResult validateSchemaAllowance(final String schemaUri) {
        final URI requestedSchemaUri = URI.create(schemaUri);

        if (!schemaProvider.isAllowedMetadataSchema(requestedSchemaUri)) {
            return ValidationResult.problem(new UnsupportedMetadataSchema(schemaUri));
        }

        return ValidationResult.ok();
    }

    private ValidationResult loadSchemaAndValidate(final Metadata metadata) {
        final URI requestedSchemaUri = URI.create(metadata.getSchema());
        final String schema = loadMetadataSchemaWithFallback(requestedSchemaUri);

        return validateMetadataAgainstSchema(metadata, schema);
    }

    private String loadMetadataSchemaWithFallback(final URI requestedSchemaUri) {
        try {
            return schemaProvider.loadMetadataSchema(requestedSchemaUri);
        } catch (Exception e) {
            final URI fallbackSchema = config.getMetadataSchemaWriteVersion();
            LOGGER.info("Schema {} is not available, using fallback {}", requestedSchemaUri, fallbackSchema);
            return schemaProvider.loadMetadataSchema(fallbackSchema);
        }
    }

    private ValidationResult validateMetadataAgainstSchema(final Metadata metadata, final String schema) {
        try {
            final String metadataJson = MAPPER.writeValueAsString(metadata);
            final JsonNode inputNode = MAPPER.readTree(metadataJson);
            return validate2020JsonSchema(schema, inputNode);
        } catch (final JsonProcessingException | SchemaNotFoundException e) {
            return ValidationResult.withErrorAndProblem(e, new MetadataSchemaViolation());
        }
    }
}
