package dev.fitko.fitconnect.core.utils;

import java.text.DecimalFormat;
import java.time.Duration;

public final class Formatter {

    private static final long BYTE = 1L;
    private static final long KB = BYTE << 10;
    private static final long MB = KB << 10;
    private static final long GB = MB << 10;

    private static final DecimalFormat DECIMAL_FORMAT = new DecimalFormat("#.##");

    private Formatter() {}

    /**
     * Formats ms to sec:ms or min:sec depending on the elapsed time.
     *
     * @param ms milliseconds that should be formatted
     * @return formatted string of the input ms
     */
    static String formatMillis(final long ms) {
        final Duration dur = Duration.ofMillis(ms);
        if (dur.getSeconds() >= 60) {
            return String.format("%02d:%02d min", dur.getSeconds() / 60, dur.getSeconds() % 60);
        } else {
            return String.format("%1d.%02d sec", dur.getSeconds(), dur.toMillis());
        }
    }

    /**
     * Formats a size in bytes to a human-readable prefix.
     *
     * @param size of the data to be formatted
     * @return size ^ 10 prefix like KB, MB or GB
     */
    public static String toHumanReadableSizePrefix(long size) {
        if (size < 0) {
            throw new IllegalArgumentException("Invalid file size: " + size);
        }
        if (size >= GB) {
            return formatSize(size, GB, "GB");
        }
        if (size >= MB) {
            return formatSize(size, MB, "MB");
        }
        if (size >= KB) {
            return formatSize(size, KB, "KB");
        }
        return formatSize(size, BYTE, "bytes");
    }

    /**
     * Shifts a byte value to an MB representation.
     *
     * @param byteValue the value to be formatted as bytes
     * @return the original byte value as MB
     */
    public static int toMegaBytes(long byteValue) {
        return (int) byteValue >> 10 >> 10;
    }

    private static String formatSize(long size, long divider, String unitName) {
        return DECIMAL_FORMAT.format((double) size / divider) + " " + unitName;
    }
}
