package dev.fitko.fitconnect.core.utils;

import static com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES;
import static dev.fitko.fitconnect.api.domain.model.event.EventIssuer.DESTINATION;
import static dev.fitko.fitconnect.api.domain.model.event.EventIssuer.SUBMISSION_SERVICE;
import static dev.fitko.fitconnect.api.domain.model.event.EventIssuer.UNDEFINED;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.jwt.SignedJWT;
import dev.fitko.fitconnect.api.domain.model.event.Event;
import dev.fitko.fitconnect.api.domain.model.event.EventClaimFields;
import dev.fitko.fitconnect.api.domain.model.event.EventIssuer;
import dev.fitko.fitconnect.api.domain.model.event.EventLogEntry;
import dev.fitko.fitconnect.api.domain.model.event.EventSubjectType;
import dev.fitko.fitconnect.api.domain.model.event.TypeAndUUID;
import dev.fitko.fitconnect.api.domain.model.event.authtags.AuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.event.problems.Problem;
import dev.fitko.fitconnect.api.exceptions.internal.EventLogException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import lombok.Data;
import lombok.NoArgsConstructor;

public final class EventLogUtil {

    private static final ObjectMapper MAPPER = new ObjectMapper().configure(FAIL_ON_UNKNOWN_PROPERTIES, false);

    public static final String AUTH_TAG_SPLIT_TOKEN = "\\.";

    private EventLogUtil() {}

    public static List<EventLogEntry> mapEventLogToEntries(final List<SignedJWT> eventLog) {
        return eventLog.stream().map(EventLogUtil::eventToLogEntry).collect(Collectors.toList());
    }

    public static EventIssuer resolveIssuerType(final String issuer) {
        if (issuer == null) {
            return UNDEFINED;
        }
        return issuer.startsWith("http") ? SUBMISSION_SERVICE : DESTINATION;
    }

    public static Event getEventFromClaims(final JWTClaimsSet claims) {
        final Map<String, Object> events = (Map) claims.getClaim(EventClaimFields.CLAIM_EVENTS);
        final Map.Entry<String, Map> eventEntry =
                (Map.Entry) events.entrySet().iterator().next();
        return Event.fromSchemaUri(eventEntry.getKey());
    }

    public static Event getEventFromJWT(final SignedJWT event) {
        try {
            final JWTClaimsSet claims = event.getJWTClaimsSet();
            final Map<String, Object> events = (Map) claims.getClaim(EventClaimFields.CLAIM_EVENTS);
            final Map.Entry<String, Map> eventEntry =
                    (Map.Entry) events.entrySet().iterator().next();
            return Event.fromSchemaUri(eventEntry.getKey());
        } catch (final ParseException e) {
            throw new EventLogException(e.getMessage(), e);
        }
    }

    public static UUID getDestinationId(final String issuer) {
        final EventIssuer issuerType = resolveIssuerType(issuer);
        if (issuerType.equals(SUBMISSION_SERVICE) || issuerType.equals(UNDEFINED)) {
            return null;
        }
        try {
            return UUID.fromString(issuer);
        } catch (final IllegalArgumentException ex) {
            throw new EventLogException("Destination id '" + issuer + "' from issuer is no valid uuid");
        }
    }

    public static EventLogEntry eventToLogEntry(final SignedJWT signedJWT) {
        final JWTClaimsSet claimsSet = getClaimsSet(signedJWT);
        final Map<String, Object> events = getClaimsAsMap(signedJWT);
        final Map.Entry<String, Object> eventEntry =
                events.entrySet().iterator().next();
        final Payload eventPayload = getEventPayload(eventEntry.getValue());
        return EventLogEntry.builder()
                .event(Event.fromSchemaUri(eventEntry.getKey()))
                .issuer(claimsSet.getIssuer())
                .issueTime(claimsSet.getIssueTime())
                .eventId(getEventId(claimsSet))
                .caseId(getTransactionClaim(claimsSet))
                .submissionId(getIdFromJWTSubject(claimsSet))
                .problems(eventPayload.getProblems())
                .build();
    }

    public static List<SignedJWT> getJWTSFromEvents(final List<String> events) {
        return events.stream().map(EventLogUtil::getJWTFromEvent).collect(Collectors.toList());
    }

    public static List<SignedJWT> getJWTSFromEvents(final List<String> events, final EventSubjectType subjectFilter) {
        return events.stream()
                .map(EventLogUtil::getJWTFromEvent)
                .filter(eventSubjectEqualsType(subjectFilter))
                .collect(Collectors.toList());
    }

    public static UUID getIdFromJWTSubject(final JWTClaimsSet claimsSet) {
        return TypeAndUUID.fromString(claimsSet.getSubject()).getUuid();
    }

    public static UUID getIdFromJWTSubject(final SignedJWT signedJWT) {
        return TypeAndUUID.fromString(getClaimsSet(signedJWT).getSubject()).getUuid();
    }

    public static AuthenticationTags getAuthTags(final SignedJWT jwt) {
        return getAuthTags(getClaimsSet(jwt));
    }

    public static AuthenticationTags getAuthTags(final JWTClaimsSet claims) {
        try {
            final Map<String, Object> eventsClaim = (Map) claims.getClaim(EventClaimFields.CLAIM_EVENTS);
            final Map.Entry<String, Map> eventEntry =
                    (Map.Entry) eventsClaim.entrySet().iterator().next();
            final Map<String, Object> events = eventEntry.getValue();
            final String authTags = MAPPER.writeValueAsString(events.get(EventClaimFields.AUTHENTICATION_TAGS));
            return MAPPER.readValue(authTags, AuthenticationTags.class);
        } catch (final JsonProcessingException e) {
            throw new EventLogException(e.getMessage(), e);
        }
    }

    public static List<SignedJWT> getFilteredJwtsFromEventLog(
            final UUID subjectFilterId, final Event event, final List<String> eventLog) {
        return getJWTSFromEvents(eventLog).stream()
                .filter(jwt -> getIdFromJWTSubject(jwt).equals(subjectFilterId))
                .filter(jwt -> getEventFromJWT(jwt).equals(event))
                .collect(Collectors.toList());
    }

    public static Predicate<SignedJWT> eventSubjectEqualsId(final UUID id) {
        return jwt -> {
            try {
                final String subject = jwt.getJWTClaimsSet().getSubject();
                return TypeAndUUID.fromString(subject).getUuid().equals(id);
            } catch (final ParseException e) {
                throw new EventLogException(e.getMessage(), e);
            }
        };
    }

    public static Predicate<SignedJWT> eventSubjectEqualsType(final EventSubjectType subjectType) {
        return jwt -> {
            try {
                final String subject = jwt.getJWTClaimsSet().getSubject();
                return TypeAndUUID.fromString(subject).getType().equals(subjectType.getName());
            } catch (final ParseException e) {
                throw new EventLogException(e.getMessage(), e);
            }
        };
    }

    private static UUID getEventId(final JWTClaimsSet claimsSet) {
        return UUID.fromString(claimsSet.getJWTID());
    }

    private static UUID getTransactionClaim(final JWTClaimsSet claimsSet) {
        try {
            return TypeAndUUID.fromString(claimsSet.getStringClaim(EventClaimFields.CLAIM_TXN))
                    .getUuid();
        } catch (final ParseException e) {
            throw new EventLogException(e.getMessage(), e);
        }
    }

    private static JWTClaimsSet getClaimsSet(final SignedJWT signedJWT) {
        try {
            return signedJWT.getJWTClaimsSet();
        } catch (final ParseException e) {
            throw new EventLogException(e.getMessage(), e);
        }
    }

    private static Map<String, Object> getClaimsAsMap(final SignedJWT signedJWT) {
        try {
            return signedJWT.getJWTClaimsSet().getJSONObjectClaim(EventClaimFields.CLAIM_EVENTS);
        } catch (final ParseException e) {
            throw new EventLogException(e.getMessage(), e);
        }
    }

    private static SignedJWT getJWTFromEvent(final String event) {
        try {
            return SignedJWT.parse(event);
        } catch (final ParseException e) {
            throw new EventLogException(e.getMessage(), e);
        }
    }

    private static Payload getEventPayload(Object payload) {
        try {
            return MAPPER.readValue(MAPPER.writeValueAsString(payload), Payload.class);
        } catch (JsonProcessingException e) {
            throw new EventLogException(e.getMessage(), e);
        }
    }

    @Data
    @NoArgsConstructor
    private static class Payload {

        @JsonProperty("authenticationTags")
        AuthenticationTags authenticationTags;

        @JsonProperty("problems")
        List<Problem> problems = new ArrayList<>();
    }
}
