package dev.fitko.fitconnect.core.submission;

import static dev.fitko.fitconnect.core.http.MimeTypes.APPLICATION_JOSE;
import static dev.fitko.fitconnect.core.http.MimeTypes.APPLICATION_JSON;

import dev.fitko.fitconnect.api.domain.limits.submission.SubmissionLimits;
import dev.fitko.fitconnect.api.domain.model.destination.PublicDestination;
import dev.fitko.fitconnect.api.domain.model.submission.AnnounceSubmission;
import dev.fitko.fitconnect.api.domain.model.submission.CreatedSubmission;
import dev.fitko.fitconnect.api.domain.model.submission.Submission;
import dev.fitko.fitconnect.api.domain.model.submission.SubmissionsForPickup;
import dev.fitko.fitconnect.api.domain.model.submission.SubmitSubmission;
import dev.fitko.fitconnect.api.exceptions.internal.RestApiException;
import dev.fitko.fitconnect.api.services.auth.OAuthService;
import dev.fitko.fitconnect.api.services.http.HttpClient;
import dev.fitko.fitconnect.api.services.submission.SubmissionService;
import dev.fitko.fitconnect.core.http.HttpHeaders;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

public class SubmissionApiService implements SubmissionService {

    public static final String SUBMISSION_PATH = "/v2/submissions/%s";
    public static final String SUBMISSIONS_PATH = "/v2/submissions";
    public static final String DESTINATION_PATH = "/v2/destinations/%s";
    public static final String SUBMISSION_ATTACHMENT_PATH = "/v2/submissions/%s/attachments/%s";
    public static final String DESTINATION_ATTACHMENT_LIMITS_PATH = "/v2/destinations/%s/limits";
    public static final String CASE_ATTACHMENT_LIMITS_PATH = "/v2/cases/%s/limits";

    private final OAuthService authService;
    private final HttpClient httpClient;
    private final String baseUrl;

    public SubmissionApiService(final OAuthService authService, final HttpClient httpClient, final String baseUrl) {
        this.authService = authService;
        this.httpClient = httpClient;
        this.baseUrl = baseUrl;
    }

    @Override
    public CreatedSubmission announceSubmission(final AnnounceSubmission submission) throws RestApiException {
        try {
            return httpClient
                    .post(
                            baseUrl + SUBMISSIONS_PATH,
                            buildHeaders(APPLICATION_JSON),
                            submission,
                            CreatedSubmission.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException(
                    "Could not announce submission for destination " + submission.getDestinationId(), e);
        }
    }

    @Override
    public void uploadAttachment(final UUID submissionId, final UUID attachmentId, final String encryptedAttachment)
            throws RestApiException {
        final String url = String.format(baseUrl + SUBMISSION_ATTACHMENT_PATH, submissionId, attachmentId);
        try {
            httpClient.put(url, buildHeaders(APPLICATION_JOSE), encryptedAttachment, Void.class);
        } catch (final RestApiException e) {
            throw new RestApiException("Could not upload attachment", e);
        }
    }

    @Override
    public void uploadAttachmentStream(
            final UUID submissionId, final UUID attachmentId, final InputStream encryptedAttachment)
            throws RestApiException {
        final String url = String.format(baseUrl + SUBMISSION_ATTACHMENT_PATH, submissionId, attachmentId);
        try {
            httpClient.put(url, buildHeaders(APPLICATION_JOSE), encryptedAttachment, Void.class);
        } catch (final RestApiException e) {
            throw new RestApiException("Could not upload attachment", e);
        }
    }

    @Override
    public String getAttachment(final UUID submissionId, final UUID attachmentId) throws RestApiException {
        final String url = String.format(baseUrl + SUBMISSION_ATTACHMENT_PATH, submissionId, attachmentId);
        try {
            return httpClient
                    .get(url, buildHeaders(APPLICATION_JOSE), String.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Could not get attachment " + attachmentId, e);
        }
    }

    @Override
    public SubmissionLimits getDestinationAttachmentLimits(UUID destinationId) throws RestApiException {
        final String url = String.format(baseUrl + DESTINATION_ATTACHMENT_LIMITS_PATH, destinationId);
        try {
            return httpClient.get(url, buildHeaders(), SubmissionLimits.class).getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Could not get attachment limits for destination " + destinationId, e);
        }
    }

    @Override
    public SubmissionLimits getCaseAttachmentLimits(UUID caseId) throws RestApiException {
        final String url = String.format(baseUrl + CASE_ATTACHMENT_LIMITS_PATH, caseId);
        try {
            return httpClient.get(url, buildHeaders(), SubmissionLimits.class).getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Could not get attachment limits for case " + caseId, e);
        }
    }

    @Override
    public PublicDestination getDestination(UUID destinationId) throws RestApiException {
        final String url = String.format(baseUrl + DESTINATION_PATH, destinationId);
        try {
            return httpClient.get(url, buildHeaders(), PublicDestination.class).getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Could not get destination for id " + destinationId, e);
        }
    }

    @Override
    public Submission sendSubmission(final SubmitSubmission submission) throws RestApiException {
        final String url = String.format(baseUrl + SUBMISSION_PATH, submission.getSubmissionId());
        try {
            return httpClient
                    .put(url, buildHeaders(APPLICATION_JSON), submission, Submission.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Could not send submission " + submission.getSubmissionId(), e);
        }
    }

    @Override
    public SubmissionsForPickup pollAvailableSubmissionsForDestination(
            final UUID destinationId, final int offset, final int limit) {
        final String urlWithQueryParams = baseUrl
                + SUBMISSIONS_PATH
                + "?destinationId="
                + destinationId
                + "&limit="
                + limit
                + "&offset="
                + offset;
        try {
            return httpClient
                    .get(urlWithQueryParams, buildHeaders(), SubmissionsForPickup.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Could not poll for available submissions on destination " + destinationId, e);
        }
    }

    @Override
    public Submission getSubmission(final UUID submissionId) throws RestApiException {
        try {
            return httpClient
                    .get(String.format(baseUrl + SUBMISSION_PATH, submissionId), buildHeaders(), Submission.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Submission with id " + submissionId + " does not exist", e, e.getStatusCode());
        }
    }

    private Map<String, String> buildHeaders(final String contentType) {
        return new HashMap<>(Map.of(
                HttpHeaders.AUTHORIZATION,
                "Bearer " + authService.getCurrentToken().getAccessToken(),
                HttpHeaders.CONTENT_TYPE,
                contentType));
    }

    private Map<String, String> buildHeaders() {
        return new HashMap<>(Map.of(
                HttpHeaders.AUTHORIZATION,
                "Bearer " + authService.getCurrentToken().getAccessToken(),
                HttpHeaders.ACCEPT,
                APPLICATION_JSON,
                HttpHeaders.ACCEPT_CHARSET,
                StandardCharsets.UTF_8.toString()));
    }
}
