package dev.fitko.fitconnect.core.routing;

import static java.util.function.Predicate.not;

import dev.fitko.fitconnect.api.domain.model.route.AreaResult;
import dev.fitko.fitconnect.api.domain.model.route.RouteResult;
import dev.fitko.fitconnect.api.exceptions.internal.RestApiException;
import dev.fitko.fitconnect.api.services.http.HttpClient;
import dev.fitko.fitconnect.api.services.routing.RoutingService;
import dev.fitko.fitconnect.core.http.HttpHeaders;
import dev.fitko.fitconnect.core.http.MimeTypes;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;

public class RoutingApiService implements RoutingService {

    public static final String ROUTING_AREA_PATH = "/v2/areas";
    public static final String ROUTING_ROUTE_PATH = "/v2/routes";

    private final HttpClient httpClient;
    private final String baseUrl;

    public RoutingApiService(final HttpClient httpClient, final String baseUrl) {
        this.httpClient = httpClient;
        this.baseUrl = baseUrl;
    }

    @Override
    public AreaResult getAreas(final List<String> searchExpressions, final int offset, final int limit)
            throws RestApiException {

        final String url = baseUrl + ROUTING_AREA_PATH;

        final List<String> queryParameters = new ArrayList<>();
        searchExpressions.forEach(search -> queryParameters.add("areaSearchexpression=" + search));
        queryParameters.add("offset=" + offset);
        queryParameters.add("limit=" + limit);
        final String queryparameterString = String.join("&", queryParameters);

        try {
            return httpClient
                    .get(
                            url + "?" + queryparameterString,
                            getHttpHeaders(MimeTypes.APPLICATION_PROBLEM_JSON),
                            AreaResult.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Area query failed", e);
        }
    }

    @Override
    public RouteResult getRoutes(
            final String leikaKey,
            final String ars,
            final String ags,
            final String areaId,
            final int offset,
            final int limit)
            throws RestApiException {

        testIfSearchCriterionIsSet(ars, ags, areaId);

        final String url = baseUrl + ROUTING_ROUTE_PATH;

        final List<String> queryParameters = new ArrayList<>();
        queryParameters.add("leikaKey=" + leikaKey);
        if (ars != null) {
            queryParameters.add("ars=" + ars);
        }
        if (ags != null) {
            queryParameters.add("ags=" + ags);
        }
        if (areaId != null) {
            queryParameters.add("areaId=" + areaId);
        }
        queryParameters.add("offset=" + offset);
        queryParameters.add("limit=" + limit);
        final String queryparameterString = String.join("&", queryParameters);

        try {
            return httpClient
                    .get(
                            url + "?" + queryparameterString,
                            getHttpHeaders(MimeTypes.APPLICATION_JSON),
                            RouteResult.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Route query failed", e);
        }
    }

    private static void testIfSearchCriterionIsSet(final String ars, final String ags, final String areaId) {
        final List<String> areaSearchCriteria = Arrays.asList(ars, ags, areaId);

        if (areaSearchCriteria.stream().allMatch(CriterionEmpty())) {
            throw new RestApiException("At least one search criterion out of ags, ars or areaId must be set");
        }

        if (areaSearchCriteria.stream().filter(not(CriterionEmpty())).count() > 1) {
            throw new RestApiException("Only one of ars, ags or areaId must be specified.");
        }
    }

    private static Predicate<String> CriterionEmpty() {
        return criterion -> criterion == null || criterion.isEmpty();
    }

    private Map<String, String> getHttpHeaders(final String mediaType) {

        return new HashMap<>(
                Map.of(HttpHeaders.ACCEPT, mediaType, HttpHeaders.ACCEPT_CHARSET, StandardCharsets.UTF_8.toString()));
    }
}
