package dev.fitko.fitconnect.core.reply;

import static dev.fitko.fitconnect.core.http.MimeTypes.APPLICATION_JOSE;
import static dev.fitko.fitconnect.core.http.MimeTypes.APPLICATION_JSON;

import dev.fitko.fitconnect.api.domain.model.event.problems.Problem;
import dev.fitko.fitconnect.api.domain.model.reply.AcceptReply;
import dev.fitko.fitconnect.api.domain.model.reply.AnnounceReply;
import dev.fitko.fitconnect.api.domain.model.reply.CreatedReply;
import dev.fitko.fitconnect.api.domain.model.reply.RejectReply;
import dev.fitko.fitconnect.api.domain.model.reply.RepliesForPickup;
import dev.fitko.fitconnect.api.domain.model.reply.Reply;
import dev.fitko.fitconnect.api.domain.model.reply.SentReply;
import dev.fitko.fitconnect.api.domain.model.reply.SubmitReply;
import dev.fitko.fitconnect.api.exceptions.internal.RestApiException;
import dev.fitko.fitconnect.api.services.auth.OAuthService;
import dev.fitko.fitconnect.api.services.http.HttpClient;
import dev.fitko.fitconnect.api.services.reply.ReplyService;
import dev.fitko.fitconnect.core.http.HttpHeaders;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

public class ReplyApiService implements ReplyService {

    public static final String REPLIES_PATH = "/v2/replies";
    public static final String REPLY_PATH = "/v2/replies/%s";
    public static final String REPLY_ACCEPT_PATH = "/v2/replies/%s/accept";
    public static final String REPLY_REJECT_PATH = "/v2/replies/%s/reject";
    public static final String REPLY_ATTACHMENT_PATH = "/v2/replies/%s/attachments/%s";

    private final OAuthService authService;
    private final HttpClient httpClient;
    private final String baseUrl;

    public ReplyApiService(final OAuthService authService, final HttpClient httpClient, final String baseUrl) {
        this.authService = authService;
        this.httpClient = httpClient;
        this.baseUrl = baseUrl;
    }

    @Override
    public CreatedReply announceReply(final AnnounceReply announceReply) throws RestApiException {
        try {
            return httpClient
                    .post(baseUrl + REPLIES_PATH, buildHeaders(APPLICATION_JSON), announceReply, CreatedReply.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Could not announce reply for case " + announceReply.getCaseId(), e);
        }
    }

    @Override
    public SentReply submitReply(final UUID replyId, final SubmitReply submitReply) throws RestApiException {
        final String url = String.format(baseUrl + REPLY_PATH, replyId);
        try {
            return httpClient
                    .put(url, buildHeaders(APPLICATION_JSON), submitReply, SentReply.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Could not submit reply " + replyId, e);
        }
    }

    @Override
    public Reply getReply(final UUID replyId) throws RestApiException {
        try {
            return httpClient
                    .get(String.format(baseUrl + REPLY_PATH, replyId), buildHeaders(), Reply.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Reply with id " + replyId + " does not exist", e, e.getStatusCode());
        }
    }

    @Override
    public RepliesForPickup getAvailableReplies(final int limit, final int offset) throws RestApiException {
        final String urlWithQueryParams = baseUrl + REPLIES_PATH + "?limit=" + limit + "&offset=" + offset;
        try {
            return httpClient
                    .get(urlWithQueryParams, buildHeaders(), RepliesForPickup.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Polling for available replies failed", e);
        }
    }

    @Override
    public String acceptReply(final UUID replyId, final AcceptReply acceptReply) {
        final String url = String.format(baseUrl + REPLY_ACCEPT_PATH, replyId);
        try {
            return httpClient
                    .put(url, buildHeaders(APPLICATION_JSON), acceptReply, String.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Sending accept reply event failed", e);
        }
    }

    @Override
    public String rejectReply(final UUID replyId, final List<Problem> problems) {
        final String url = String.format(baseUrl + REPLY_REJECT_PATH, replyId);
        try {
            return httpClient
                    .put(url, buildHeaders(APPLICATION_JSON), new RejectReply(problems), String.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Sending reject reply event failed", e);
        }
    }

    @Override
    public void uploadAttachment(final UUID replyId, final UUID attachmentId, final String encryptedAttachment)
            throws RestApiException {
        final String url = String.format(baseUrl + REPLY_ATTACHMENT_PATH, replyId, attachmentId);
        try {
            httpClient.put(url, buildHeaders(APPLICATION_JOSE), encryptedAttachment, Void.class);
        } catch (final RestApiException e) {
            throw new RestApiException("Could not upload attachment for reply " + replyId, e);
        }
    }

    @Override
    public String getAttachment(final UUID replyId, final UUID attachmentId) throws RestApiException {
        final String url = String.format(baseUrl + REPLY_ATTACHMENT_PATH, replyId, attachmentId);
        try {
            return httpClient
                    .get(url, buildHeaders(APPLICATION_JOSE), String.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Could not get attachment " + attachmentId + " for reply " + replyId, e);
        }
    }

    private Map<String, String> buildHeaders(final String contentType) {
        return new HashMap<>(Map.of(
                HttpHeaders.AUTHORIZATION,
                "Bearer " + authService.getCurrentToken().getAccessToken(),
                HttpHeaders.CONTENT_TYPE,
                contentType));
    }

    private Map<String, String> buildHeaders() {
        return new HashMap<>(Map.of(
                HttpHeaders.AUTHORIZATION,
                "Bearer " + authService.getCurrentToken().getAccessToken(),
                HttpHeaders.ACCEPT,
                APPLICATION_JSON,
                HttpHeaders.ACCEPT_CHARSET,
                StandardCharsets.UTF_8.toString()));
    }
}
