package dev.fitko.fitconnect.core.http.ssl;

import dev.fitko.fitconnect.api.config.ZBPCertConfig;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectInitialisationException;
import java.io.ByteArrayInputStream;
import java.nio.charset.StandardCharsets;
import java.security.KeyFactory;
import java.security.KeyStore;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.security.spec.PKCS8EncodedKeySpec;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.NoArgsConstructor;
import okhttp3.internal.platform.Platform;

@NoArgsConstructor(access = AccessLevel.PRIVATE)
public class SSLContextBuilder {

    public static HttpClientSSLContext build(ZBPCertConfig config) {
        try {
            final TrustManager[] trustManagers =
                    createTrustManager(config); // determines whether we accept the server cert
            final KeyManager[] keyManagers = createIdentityStore(config); // determines whether the server accepts us

            final SSLContext sslContext = SSLContext.getInstance("TLS");
            sslContext.init(keyManagers, trustManagers, null);
            SSLSocketFactory sslSocketFactory = sslContext.getSocketFactory();
            return new HttpClientSSLContext(sslSocketFactory, (X509TrustManager) trustManagers[0]);
        } catch (Exception e) {
            throw new FitConnectInitialisationException(e.getMessage(), e);
        }
    }

    private static TrustManager[] createTrustManager(ZBPCertConfig config) throws Exception {
        if (config.getZbpServerCertificate().isEmpty()) {
            // main use case: we trust the CAs present on the system
            return new TrustManager[] {Platform.get().platformTrustManager()};
        }
        // otherwise, if a server cert is set explicitly, we only trust that single cert
        final CertificateFactory certificateFactory = CertificateFactory.getInstance("X.509");
        final ByteArrayInputStream serverCertStream =
                new ByteArrayInputStream(config.getZbpServerCertificate().get().getBytes(StandardCharsets.UTF_8));
        final Certificate trustedCertificate = certificateFactory.generateCertificate(serverCertStream);
        final KeyStore trustStore = createEmptyKeyStore();
        trustStore.setCertificateEntry("server-certificate", trustedCertificate);
        final TrustManagerFactory trustManagerFactory =
                TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
        trustManagerFactory.init(trustStore);
        return trustManagerFactory.getTrustManagers();
    }

    private static KeyManager[] createIdentityStore(ZBPCertConfig config) throws Exception {
        final CertificateFactory certificateFactory = CertificateFactory.getInstance("X.509");
        final KeyFactory keyFactory = KeyFactory.getInstance("RSA");
        final byte[] privateKeyAsBytes =
                config.getClientPrivateKey().toRSAPrivateKey().getEncoded();

        final PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(privateKeyAsBytes);
        final ByteArrayInputStream clientCertStream =
                new ByteArrayInputStream(config.getClientCertificate().getBytes());
        final Certificate certificateChain = certificateFactory.generateCertificate(clientCertStream);

        final KeyStore identityStore = createEmptyKeyStore();
        identityStore.setKeyEntry(
                "client", keyFactory.generatePrivate(keySpec), null, new Certificate[] {certificateChain});

        final KeyManagerFactory keyManagerFactory =
                KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
        keyManagerFactory.init(identityStore, null);
        return keyManagerFactory.getKeyManagers();
    }

    private static KeyStore createEmptyKeyStore() throws Exception {
        KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
        keyStore.load(null, null);
        return keyStore;
    }

    @Getter
    @AllArgsConstructor
    public static class HttpClientSSLContext {
        SSLSocketFactory sslSocketFactory;
        X509TrustManager trustManager;
    }
}
