package dev.fitko.fitconnect.core.http.retrylogic;

import static dev.fitko.fitconnect.core.utils.Preconditions.checkArgumentAndThrow;

import java.util.Random;
import lombok.Getter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** A backoff-strategy that calculates a delay time until processing can continue. */
@Getter
public class ExponentialBackoff {

    private static final Logger LOGGER = LoggerFactory.getLogger(ExponentialBackoff.class);

    private final int maxRetries;
    private final int initialDelay;
    private int retriesLeft;
    private int delay;

    public ExponentialBackoff(int maxRetries, int initialDelayInMs) {
        checkArgumentAndThrow(maxRetries <= 0, "maxRetries must be > 0");
        checkArgumentAndThrow(initialDelayInMs <= 0, "initialDelayInMs must be > 0");
        this.maxRetries = maxRetries;
        this.initialDelay = initialDelayInMs;
        this.retriesLeft = maxRetries;
        this.delay = initialDelayInMs;
    }

    public ExponentialBackoff(int maxRetries) {
        this(maxRetries, new Random().nextInt(1000));
    }

    /**
     * Get the retries that are currently available.
     *
     * @return remaining retries > 0
     * @see #getMaxRetries()
     */
    public boolean retriesAvailable() {
        return retriesLeft > 0;
    }

    /**
     * Call will delay the current thread by exponential backoff starting with the initial delay. The
     * delay doubles on each call.
     */
    public void backoffWithDelay() {
        try {
            Thread.sleep(delay);
        } catch (InterruptedException e) {
            LOGGER.error("Waiting for next request attempt failed", e);
        }
        if (retriesAvailable()) {
            retriesLeft--;
            delay *= 2;
        }
    }
}
