package dev.fitko.fitconnect.core.http.interceptors;

import static dev.fitko.fitconnect.core.http.HttpHeaders.USER_AGENT;

import dev.fitko.fitconnect.api.config.resources.BuildInfo;
import dev.fitko.fitconnect.core.utils.Strings;
import java.io.IOException;
import okhttp3.Interceptor;
import okhttp3.Request;
import okhttp3.Response;

public class UserAgentInterceptor implements Interceptor {

    private static final String OS_NAME = System.getProperty("os.name");
    private static final String JAVA_VERSION = System.getProperty("java.version");
    private static final String USER_AGENT_TEMPLATE = "%s/%s (os:%s;java:%s;commit:%s)";
    private static final String COMMIT_ID_TEMPLATE = "${git.commit.id.full}";

    private final String productName;
    private final String productVersion;
    private final String commit;

    public UserAgentInterceptor(final BuildInfo buildInfo) {
        productName = buildInfo.getProductName();
        productVersion = buildInfo.getProductVersion();
        commit = getCommitId(buildInfo);
    }

    @Override
    public Response intercept(final Chain chain) throws IOException {

        Request request = chain.request()
                .newBuilder()
                .header(
                        USER_AGENT,
                        String.format(USER_AGENT_TEMPLATE, productName, productVersion, OS_NAME, JAVA_VERSION, commit))
                .build();

        return chain.proceed(request);
    }

    private static String getCommitId(BuildInfo buildInfo) {
        final String commitId = buildInfo.getCommit();
        if (Strings.isNullOrEmpty(commitId) || commitId.equals(COMMIT_ID_TEMPLATE)) {
            return "Not available";
        }
        return buildInfo.getCommit();
    }
}
