package dev.fitko.fitconnect.core.http.interceptors;

import static java.lang.Math.round;

import dev.fitko.fitconnect.api.config.http.RetryConfig;
import dev.fitko.fitconnect.api.exceptions.internal.RestApiException;
import dev.fitko.fitconnect.core.http.retrylogic.ExponentialBackoff;
import dev.fitko.fitconnect.core.http.retrylogic.RandomValue;
import java.io.IOException;
import lombok.AllArgsConstructor;
import lombok.Getter;
import okhttp3.Interceptor;
import okhttp3.Request;
import okhttp3.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Intercepts HTTP-calls and retries with delay based on exponential backoff.
 *
 * @see ExponentialBackoff
 */
public class RetryInterceptor implements Interceptor {

    private static final Logger LOGGER = LoggerFactory.getLogger(RetryInterceptor.class);
    public static final String EXCEPTION_MSG =
            "Request failed with exception, retrying with a delay of {}ms, remaining attempts {}";
    public static final String UNSUCCESSFUL_MSG =
            "Request failed with status {}, retrying with a delay of {}ms, remaining attempts {}";
    private final RetryConfig retryConfig;

    public RetryInterceptor(RetryConfig retryConfig) {
        this.retryConfig = retryConfig;
    }

    @Override
    public Response intercept(Chain chain) {

        PossibleResponse possibleResponse = attemptNextResponse(chain);

        // continue if response is successful
        if (possibleResponse.isSuccessful()) {
            return possibleResponse.getActualResponse();
        }

        final ExponentialBackoff backoffHandler = getBackoffHandler();

        // retry n-times with backoff delay
        while (shouldRetry(possibleResponse, backoffHandler)) {
            backoffHandler.backoffWithDelay();
            closePreviousResponse(possibleResponse);
            possibleResponse = attemptNextResponse(chain);
        }

        if (!backoffHandler.retriesAvailable() && possibleResponse.isNotSuccessful()) {
            LOGGER.error("*** Retries consumed after {} attempts ***", backoffHandler.getMaxRetries());
        }

        // rethrow original exception if present
        if (possibleResponse.hasException()) {
            final Exception exception = possibleResponse.getException();
            throw new RestApiException(exception.getMessage(), exception);
        }

        if (possibleResponse.isSuccessful()) {
            LOGGER.info("*** Retry was successful ! ***");
        }

        return possibleResponse.getActualResponse();
    }

    private void closePreviousResponse(PossibleResponse response) {
        if (response.hasResponse()) {
            response.getActualResponse().close();
        }
    }

    private PossibleResponse attemptNextResponse(Chain chain) {
        try {
            final Request request = chain.request();
            return PossibleResponse.withResponse(chain.proceed(request));
        } catch (IOException e) {
            LOGGER.debug(e.getMessage(), e);
            return PossibleResponse.withException(e);
        }
    }

    private boolean shouldRetry(PossibleResponse response, ExponentialBackoff backoffHandler) {

        final long backoffDelay = backoffHandler.getDelay();
        final int leftRetries = backoffHandler.getRetriesLeft() - 1;
        final boolean retriesAvailable = backoffHandler.retriesAvailable();

        if (response.hasException() && retriesAvailable) {
            LOGGER.warn(EXCEPTION_MSG, backoffDelay, leftRetries);
            return true;
        } else if (response.hasResponse() && statusCodeRetryable(response) && retriesAvailable) {
            LOGGER.warn(UNSUCCESSFUL_MSG, response.getActualResponse().code(), backoffDelay, leftRetries);
            return true;
        } else {
            return false;
        }
    }

    private boolean statusCodeRetryable(PossibleResponse response) {
        return retryConfig
                .getRetryableStatusCodes()
                .contains(response.getActualResponse().code());
    }

    private ExponentialBackoff getBackoffHandler() {
        // create start delay in bounds to distribute request times
        int min = round(retryConfig.getInitialDelayInMs() * 0.7f);
        int max = round(retryConfig.getInitialDelayInMs() * 1.3f);
        int distributedStartDelay = RandomValue.randomIntInRange(min, max);
        return new ExponentialBackoff(retryConfig.getMaxRetryCount(), distributedStartDelay);
    }

    @Getter
    @AllArgsConstructor
    private static final class PossibleResponse {
        private final Response actualResponse;
        private final Exception exception;

        static PossibleResponse withResponse(Response response) {
            return new PossibleResponse(response, null);
        }

        static PossibleResponse withException(Exception exception) {
            return new PossibleResponse(null, exception);
        }

        boolean isSuccessful() {
            return hasResponse() && actualResponse.isSuccessful();
        }

        boolean isNotSuccessful() {
            return !isSuccessful() || hasException();
        }

        boolean hasException() {
            return exception != null && actualResponse == null;
        }

        boolean hasResponse() {
            return actualResponse != null;
        }
    }
}
