package dev.fitko.fitconnect.core.http;

import static com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES;
import static dev.fitko.fitconnect.client.util.MetadataDeserializationHelper.deserializeMetadata;
import static java.nio.charset.StandardCharsets.UTF_8;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import dev.fitko.fitconnect.api.config.ZBPCertConfig;
import dev.fitko.fitconnect.api.config.defaults.ZBPEnvironment;
import dev.fitko.fitconnect.api.config.http.HttpConfig;
import dev.fitko.fitconnect.api.config.http.ProxyAuth;
import dev.fitko.fitconnect.api.config.http.ProxyConfig;
import dev.fitko.fitconnect.api.config.http.Timeouts;
import dev.fitko.fitconnect.api.domain.model.metadata.Metadata;
import dev.fitko.fitconnect.api.exceptions.internal.RestApiException;
import dev.fitko.fitconnect.api.services.http.HttpClient;
import dev.fitko.fitconnect.api.services.http.HttpResponse;
import dev.fitko.fitconnect.core.http.ssl.SSLContextBuilder;
import dev.fitko.fitconnect.core.utils.Preconditions;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.Proxy;
import java.nio.charset.Charset;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import okhttp3.*;
import okio.BufferedSink;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultHttpClient implements HttpClient {

    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultHttpClient.class);
    private static final ObjectMapper MAPPER =
            new ObjectMapper().configure(FAIL_ON_UNKNOWN_PROPERTIES, false).registerModule(new JavaTimeModule());
    private final OkHttpClient httpClient;

    public DefaultHttpClient() {
        this(new HttpConfig(), Collections.emptyList());
    }

    public DefaultHttpClient(HttpConfig httpConfig, List<Interceptor> interceptors) {
        this(httpConfig, interceptors, null, null);
    }

    public DefaultHttpClient(
            HttpConfig httpConfig,
            List<Interceptor> interceptors,
            ZBPCertConfig zbpCertConfig,
            ZBPEnvironment zbpEnvironment) {
        var builder = new OkHttpClient.Builder();
        setInterceptors(interceptors, builder);
        setTimeouts(httpConfig.getTimeouts(), builder);
        setProxy(httpConfig.getProxyConfig(), builder);
        setSSLContext(zbpCertConfig, zbpEnvironment, builder);
        httpClient = builder.build();
    }

    private void setSSLContext(
            ZBPCertConfig zbpCertConfig, ZBPEnvironment zbpEnvironment, OkHttpClient.Builder builder) {
        if (zbpCertConfig != null && zbpEnvironment != null && !zbpEnvironment.isSendCertAsCookie()) {
            addSSLContext(zbpCertConfig, builder);
        }
    }

    private void setProxy(final ProxyConfig proxyConfig, final OkHttpClient.Builder builder) {
        if (!proxyConfig.isProxySet()) {
            LOGGER.info("Creating HttpClient without proxy configuration.");
            return;
        }
        final Proxy proxy = proxyConfig.getHttpProxy();
        LOGGER.info("Creating HttpClient with proxy configuration: {}", proxy);
        builder.proxy(proxy);
        if (proxyConfig.hasBasicAuthentication()) {
            LOGGER.info("Creating proxy with basic authentication");
            final Authenticator proxyAuthenticator = getProxyAuthenticator(proxyConfig.getBasicAuth());
            builder.proxyAuthenticator(proxyAuthenticator);
        }
    }

    private Authenticator getProxyAuthenticator(ProxyAuth auth) {
        return (route, response) -> {
            String credential = Credentials.basic(auth.getUsername(), auth.getPassword());
            return response.request()
                    .newBuilder()
                    .header("Proxy-Authorization", credential)
                    .build();
        };
    }

    private void setInterceptors(final List<Interceptor> interceptors, final OkHttpClient.Builder builder) {
        interceptors.forEach(builder::addInterceptor);
    }

    private void setTimeouts(final Timeouts timeouts, final OkHttpClient.Builder builder) {
        builder.callTimeout(timeouts.getCallTimeoutInSeconds());
        builder.readTimeout(timeouts.getReadTimeoutInSeconds());
        builder.writeTimeout(timeouts.getWriteTimeoutInSeconds());
        builder.connectTimeout(timeouts.getConnectionTimeoutInSeconds());
    }

    @Override
    public <R> HttpResponse<R> get(final String url, final Map<String, String> headers, final Class<R> responseType) {

        final Request request = new Request.Builder()
                .url(url)
                .get()
                .headers(Headers.of(headers))
                .build();

        try (final Response response = httpClient.newCall(request).execute()) {
            return evaluateStatusAndRespond(response, responseType);
        } catch (final RestApiException restApiException) {
            // rethrow to keep the status code
            throw restApiException;
        } catch (Exception exception) {
            throw new RestApiException("HTTP GET call to '" + url + "' failed.", exception);
        }
    }

    @Override
    public <P, R> HttpResponse<R> post(
            final String url, final Map<String, String> headers, final P httpPayload, final Class<R> responseType) {
        try {
            final RequestBody requestBody = createRequestBody(headers, httpPayload);
            final Request request = new Request.Builder()
                    .url(url)
                    .post(requestBody)
                    .headers(Headers.of(headers))
                    .build();

            try (final Response response = httpClient.newCall(request).execute()) {
                return evaluateStatusAndRespond(response, responseType);
            }

        } catch (final RestApiException restApiException) {
            // rethrow to keep the status code
            throw restApiException;
        } catch (Exception exception) {
            throw new RestApiException("HTTP POST call to '" + url + "' failed.", exception);
        }
    }

    @Override
    public <P, R> HttpResponse<R> put(
            final String url, final Map<String, String> headers, final P httpPayload, final Class<R> responseType) {

        try {
            final RequestBody requestBody = createRequestBody(headers, httpPayload);
            final Request request = new Request.Builder()
                    .url(url)
                    .put(requestBody)
                    .headers(Headers.of(headers))
                    .build();

            try (final Response response = httpClient.newCall(request).execute()) {
                return evaluateStatusAndRespond(response, responseType);
            }

        } catch (final RestApiException restApiException) {
            // rethrow to keep the status code
            throw restApiException;
        } catch (Exception exception) {
            throw new RestApiException("HTTP PUT call to '" + url + "' failed.", exception);
        }
    }

    @Override
    public <P, R> HttpResponse<R> patch(
            final String url, final Map<String, String> headers, final P httpPayload, final Class<R> responseType) {
        try {
            final RequestBody requestBody = createRequestBody(headers, httpPayload);
            final Request request = new Request.Builder()
                    .url(url)
                    .patch(requestBody)
                    .headers(Headers.of(headers))
                    .build();

            try (final Response response = httpClient.newCall(request).execute()) {
                return evaluateStatusAndRespond(response, responseType);
            }

        } catch (final IOException exception) {
            throw new RestApiException("HTTP PATCH call to '" + url + "' failed.", exception);
        }
    }

    @Override
    public HttpResponse<Void> delete(final String url, final Map<String, String> headers) {
        try {
            final Request request = new Request.Builder()
                    .url(url)
                    .delete()
                    .headers(Headers.of(headers))
                    .build();

            try (final Response response = httpClient.newCall(request).execute()) {
                return evaluateStatusAndRespond(response, Void.class);
            }

        } catch (final IOException exception) {
            throw new RestApiException("HTTP DELETE call to '" + url + "' failed.", exception);
        }
    }

    /**
     * The methods {@link ObjectMapper#writeValueAsBytes(Object)} and {@link
     * ObjectMapper#writeValueAsString(Object)} seem to have issues with writing a raw string into a
     * new string, resulting in obsolete quotation marks at begin and end of the actual string.
     * Therefore, this method contains a dedicated logic part for dealing with raw strings.
     *
     * <p>Also, we have to do a little workaround with {@link String#getBytes(Charset)} due to OkHttp
     * enforcing a standard value for the header "Content-Type" in case of raw strings. See <a
     * href="https://github.com/square/okhttp/issues/2099">https://github.com/square/okhttp/issues/2099</a>
     * for more information on that issue.
     */
    <T> RequestBody createRequestBody(final Map<String, String> headers, final T httpPayload)
            throws JsonProcessingException {

        if (httpPayload instanceof MultipartBody) {
            return (MultipartBody) httpPayload;
        }

        if (httpPayload instanceof InputStream) {
            return getRequestBodyWithStreamPayload(headers, (InputStream) httpPayload);
        }

        return RequestBody.create(buildPayloadBody(httpPayload), getContentType(headers));
    }

    <R> HttpResponse<R> evaluateStatusAndRespond(final Response response, final Class<R> responseType)
            throws IOException {

        if (!response.isSuccessful()) {
            final String message = response.body() != null ? response.body().string() : "HTTP call failed.";
            throw new RestApiException(message, response.code());
        }
        final ResponseBody responseBody = response.body();
        if (responseBody == null) {
            throw new RestApiException("Response body is null.", response.code());
        }

        if (responseType.equals(Void.class)) {
            return new HttpResponse<>(response.code(), null);
        }

        if (responseType.equals(String.class)) {
            return new HttpResponse<>(response.code(), (R) responseBody.string());
        }

        if (responseType.equals(InputStream.class)) {
            var buffer = new ByteArrayInputStream(response.body().byteStream().readAllBytes());
            return new HttpResponse<>(response.code(), (R) buffer);
        }

        if (responseType.equals(Metadata.class)) {
            return new HttpResponse<>(response.code(), (R)
                    deserializeMetadata(MAPPER, responseBody.string().getBytes(UTF_8)));
        }

        return new HttpResponse<>(response.code(), MAPPER.readValue(responseBody.string(), responseType));
    }

    private void addSSLContext(ZBPCertConfig config, OkHttpClient.Builder clientBuilder) {
        Preconditions.checkArgumentAndThrow(!config.hasValidConfiguration(), "Required ZBP certificates are not set");
        var sslContext = SSLContextBuilder.build(config);
        clientBuilder.sslSocketFactory(sslContext.getSslSocketFactory(), sslContext.getTrustManager());
    }

    private static RequestBody getRequestBodyWithStreamPayload(
            final Map<String, String> headers, final InputStream httpPayload) {
        return new RequestBody() {
            @Override
            public MediaType contentType() {
                return MediaType.parse(headers.get(HttpHeaders.CONTENT_TYPE));
            }

            @Override
            public void writeTo(final BufferedSink sink) throws IOException {
                httpPayload.transferTo(sink.outputStream());
            }
        };
    }

    private static MediaType getContentType(final Map<String, String> headers) {
        if (headers.containsKey(HttpHeaders.CONTENT_TYPE)) {
            return MediaType.parse(headers.get(HttpHeaders.CONTENT_TYPE));
        }
        return MediaType.get(MimeTypes.APPLICATION_JSON);
    }

    private static <T> byte[] buildPayloadBody(final T httpPayload) throws JsonProcessingException {
        if (httpPayload instanceof String) {
            return ((String) httpPayload).getBytes(UTF_8);
        }
        return MAPPER.writeValueAsBytes(httpPayload);
    }
}
