package dev.fitko.fitconnect.core.destination;

import static dev.fitko.fitconnect.core.http.MimeTypes.APPLICATION_JSON;

import dev.fitko.fitconnect.api.domain.limits.destination.DestinationLimits;
import dev.fitko.fitconnect.api.domain.limits.destination.LimitChangeRequest;
import dev.fitko.fitconnect.api.domain.model.destination.CreateDestination;
import dev.fitko.fitconnect.api.domain.model.destination.Destination;
import dev.fitko.fitconnect.api.domain.model.destination.Destinations;
import dev.fitko.fitconnect.api.domain.model.jwk.ApiJwk;
import dev.fitko.fitconnect.api.domain.model.jwk.ApiJwks;
import dev.fitko.fitconnect.api.exceptions.internal.RestApiException;
import dev.fitko.fitconnect.api.services.auth.OAuthService;
import dev.fitko.fitconnect.api.services.destination.DestinationService;
import dev.fitko.fitconnect.api.services.http.HttpClient;
import dev.fitko.fitconnect.core.http.HttpHeaders;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

public class DestinationApiService implements DestinationService {

    private static final String DESTINATIONS_PATH = "/v2/destinations";
    private static final String DESTINATION_PATH = "/v2/destinations/%s";
    private static final String DESTINATION_KEYS_PATH = "/v2/destinations/%s/keys";
    private static final String DESTINATION_KEY_PATH = "/v2/destinations/%s/keys/%s";
    private static final String DESTINATION_ATTACHMENT_LIMITS_PATH = "/v2/destinations/%s/limits";

    private final OAuthService authService;
    private final HttpClient httpClient;
    private final String baseUrl;

    public DestinationApiService(final OAuthService authService, final HttpClient httpClient, final String baseUrl) {
        this.authService = authService;
        this.httpClient = httpClient;
        this.baseUrl = baseUrl;
    }

    @Override
    public Destination getDestination(final UUID destinationId) throws RestApiException {
        final String urlWithQueryParams = String.format(baseUrl + DESTINATION_PATH, destinationId);
        try {
            return httpClient
                    .get(urlWithQueryParams, buildHeaders(), Destination.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Could not get destination " + destinationId, e);
        }
    }

    @Override
    public Destination createDestination(final CreateDestination destination) {
        final String url = baseUrl + DESTINATIONS_PATH;
        try {
            return httpClient
                    .post(url, buildHeaders(APPLICATION_JSON), destination, Destination.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Could not create new destination", e);
        }
    }

    @Override
    public Destination updateDestination(final Destination destination) {
        final String urlWithQueryParams = String.format(baseUrl + DESTINATION_PATH, destination.getDestinationId());
        try {
            return httpClient
                    .put(urlWithQueryParams, buildHeaders(APPLICATION_JSON), destination, Destination.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Could not update destination " + destination.getDestinationId(), e);
        }
    }

    @Override
    public void deleteDestination(final UUID destinationId) {
        final String urlWithQueryParams = String.format(baseUrl + DESTINATION_PATH, destinationId);
        try {
            httpClient.delete(urlWithQueryParams, buildHeaders(APPLICATION_JSON));
        } catch (final RestApiException e) {
            throw new RestApiException("Could not delete destination " + destinationId, e);
        }
    }

    @Override
    public Destinations listDestinations(final int offset, final int limit) {
        final String urlWithQueryParams = baseUrl + DESTINATIONS_PATH + "?limit=" + limit + "&offset=" + offset;
        try {
            return httpClient
                    .get(urlWithQueryParams, buildHeaders(), Destinations.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Could not poll for available destinations", e);
        }
    }

    @Override
    public DestinationLimits getAttachmentLimits(final UUID destinationId) {
        final String urlWithQueryParams = String.format(baseUrl + DESTINATION_ATTACHMENT_LIMITS_PATH, destinationId);
        try {
            return httpClient
                    .get(urlWithQueryParams, buildHeaders(), DestinationLimits.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Could not get attachment limits for destination " + destinationId, e);
        }
    }

    @Override
    public DestinationLimits requestLimitChange(UUID destinationId, LimitChangeRequest limitChangeRequest) {
        final String urlWithQueryParams = String.format(baseUrl + DESTINATION_ATTACHMENT_LIMITS_PATH, destinationId);
        try {
            final DestinationLimits destinationLimits = DestinationLimits.withChangeRequest(limitChangeRequest);
            return httpClient
                    .patch(
                            urlWithQueryParams,
                            buildHeaders(APPLICATION_JSON),
                            destinationLimits,
                            DestinationLimits.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Could not send limit change request for destination " + destinationId, e);
        }
    }

    @Override
    public ApiJwks listKeys(final UUID destinationId, final int offset, final int limit) {
        final String urlWithQueryParams =
                String.format(baseUrl + DESTINATION_KEYS_PATH, destinationId) + "?limit=" + limit + "&offset=" + offset;
        try {
            return httpClient
                    .get(urlWithQueryParams, buildHeaders(), ApiJwks.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Could not list keys for destination " + destinationId, e);
        }
    }

    @Override
    public ApiJwk getKey(final UUID destinationId, final String keyId) {
        final String urlWithQueryParams = String.format(baseUrl + DESTINATION_KEY_PATH, destinationId, keyId);
        try {
            return httpClient
                    .get(urlWithQueryParams, buildHeaders(), ApiJwk.class)
                    .getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Could not get key " + keyId + " for destination " + destinationId, e);
        }
    }

    @Override
    public void addKey(final UUID destinationId, ApiJwk apiJwk) {
        final String urlWithQueryParams = String.format(baseUrl + DESTINATION_KEYS_PATH, destinationId);
        try {
            httpClient.post(urlWithQueryParams, buildHeaders(APPLICATION_JSON), apiJwk, Void.class);
        } catch (final RestApiException e) {
            throw new RestApiException("Could not add key " + apiJwk.getKid() + " to destination " + destinationId, e);
        }
    }

    private Map<String, String> buildHeaders(final String contentType) {
        return new HashMap<>(Map.of(
                HttpHeaders.AUTHORIZATION,
                "Bearer " + authService.getCurrentToken().getAccessToken(),
                HttpHeaders.CONTENT_TYPE,
                contentType));
    }

    private Map<String, String> buildHeaders() {
        return new HashMap<>(Map.of(
                HttpHeaders.AUTHORIZATION,
                "Bearer " + authService.getCurrentToken().getAccessToken(),
                HttpHeaders.ACCEPT,
                APPLICATION_JSON,
                HttpHeaders.ACCEPT_CHARSET,
                StandardCharsets.UTF_8.toString()));
    }
}
