package dev.fitko.fitconnect.core.crypto.utils;

import com.nimbusds.jose.Algorithm;
import com.nimbusds.jose.jwk.JWK;
import com.nimbusds.jose.jwk.KeyOperation;
import com.nimbusds.jose.jwk.RSAKey;
import com.nimbusds.jose.util.Base64;
import dev.fitko.fitconnect.api.exceptions.internal.KeyGenerationException;
import java.security.InvalidParameterException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.interfaces.RSAPublicKey;
import java.util.Collections;
import java.util.List;
import java.util.Set;

/** Generates {@link JWK}s and {@link KeyPair}s. */
public final class KeyGenerator {

    private KeyGenerator() {}

    /**
     * Generate a set of RSA public and private key.
     *
     * @param keySize modulus length in bits
     * @return pair of public and private key
     */
    public static KeyPair buildRSAKeyPair(final int keySize) {
        try {
            final KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance("RSA");
            keyPairGenerator.initialize(keySize);
            return keyPairGenerator.generateKeyPair();
        } catch (final NoSuchAlgorithmException | InvalidParameterException e) {
            throw new KeyGenerationException(e.getMessage(), e);
        }
    }

    /**
     * Build a JWK with the given parameters with x5c keychain.
     *
     * @param keyPair pair of public and private key
     * @param keyId unique identifier
     * @param keyOperation key operation that determines the usage of the key
     * @param algorithm encryption algorithm
     * @param x5cCertificateChain x5c chain of the certificate
     * @return JWK
     */
    public static JWK buildJWK(
            final KeyPair keyPair,
            final String keyId,
            final KeyOperation keyOperation,
            final Algorithm algorithm,
            List<Base64> x5cCertificateChain) {
        final RSAKey.Builder keyBuilder = new RSAKey.Builder((RSAPublicKey) keyPair.getPublic())
                .privateKey(keyPair.getPrivate())
                .keyID(keyId)
                .keyOperations(Set.of(keyOperation))
                .algorithm(algorithm);

        if (!x5cCertificateChain.isEmpty()) {
            keyBuilder.x509CertChain(x5cCertificateChain);
        }

        return keyBuilder.build();
    }

    /**
     * Build a JWK with the given parameters without x5c keychain.
     *
     * @param keyPair pair of public and private key
     * @param keyId unique identifier
     * @param keyOperation key operation that determines the usage of the key
     * @param algorithm encryption algorithm
     * @return JWK
     */
    public static JWK buildJWK(
            final KeyPair keyPair, final String keyId, final KeyOperation keyOperation, final Algorithm algorithm) {
        return buildJWK(keyPair, keyId, keyOperation, algorithm, Collections.emptyList());
    }
}
