package dev.fitko.fitconnect.core.crypto.utils;

import com.nimbusds.jose.util.Base64;
import com.nimbusds.jose.util.X509CertUtils;
import dev.fitko.fitconnect.api.exceptions.internal.CertificateEncodingException;
import dev.fitko.fitconnect.api.exceptions.internal.KeyGenerationException;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.security.KeyPair;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.time.Instant;
import java.util.Date;
import javax.security.auth.x500.X500Principal;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;
import org.bouncycastle.asn1.x500.RDN;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.asn1.x500.style.IETFUtils;
import org.bouncycastle.cert.X509v3CertificateBuilder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cert.jcajce.JcaX509v3CertificateBuilder;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;

/**
 * Util class to generate X.509 certificates for testing purposes and Base64 encode a certificate.
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public final class CertUtils {

    public static final String PEM_BEGIN_MARKER = "-----BEGIN CERTIFICATE-----";
    public static final String PEM_END_MARKER = "-----END CERTIFICATE-----";

    /**
     * Generate a self-signed X509Certificate for usage in test keys. This certificate must not be
     * used in productive environments.
     *
     * @param keyPair the keypair of public and private key used for the certificate
     * @param validityFromNow duration of the validity from now
     * @return the self-signed certificate
     * @throws KeyGenerationException if the certificate generation failed
     * @see KeyGenerator#buildRSAKeyPair(int)
     */
    public static X509Certificate generateX509Certificate(final KeyPair keyPair, Duration validityFromNow)
            throws KeyGenerationException {
        final Instant now = Instant.now();
        final Date notBefore = Date.from(now);
        final Date notAfter = Date.from(now.plus(validityFromNow));
        try {
            final ContentSigner contentSigner =
                    new JcaContentSignerBuilder("SHA512withRSA").build(keyPair.getPrivate());
            final X500Name x500Name = new X500Name("CN=FIT-Connect");
            final X500Name x500Subject = new X500Name("C=TestCert");
            final BigInteger serialNr = BigInteger.valueOf(now.toEpochMilli());
            final X509v3CertificateBuilder certificateBuilder = new JcaX509v3CertificateBuilder(
                    x500Name, serialNr, notBefore, notAfter, x500Subject, keyPair.getPublic());
            return new JcaX509CertificateConverter().getCertificate(certificateBuilder.build(contentSigner));
        } catch (final CertificateException | OperatorCreationException e) {
            throw new KeyGenerationException(e.getMessage(), e);
        }
    }

    /**
     * Converts an X.509 certificate to a Base64 encoded string.
     *
     * @param certificateData the certificate data as byte array
     * @return Base64 encoded certificate as string
     * @throws CertificateEncodingException if the certificate could not be parsed or encoded
     */
    public static String convertToBase64EncodedX509Certificate(final byte[] certificateData)
            throws CertificateEncodingException {
        try (InputStream certificateInputStream = new ByteArrayInputStream(certificateData)) {
            final CertificateFactory certificateFactory = CertificateFactory.getInstance("X.509");
            final X509Certificate x509Certificate =
                    (X509Certificate) certificateFactory.generateCertificate(certificateInputStream);
            return Base64.encode(x509Certificate.getEncoded()).toString();
        } catch (final CertificateException | IOException e) {
            throw new CertificateEncodingException("Failed to encode certificate", e);
        }
    }

    /**
     * Extracts the subject common name from a given certificate.
     *
     * @param pemCertificate the cert in pem format
     * @return subject as string
     */
    public static String getSubjectCNFromCertificate(String pemCertificate) {
        try {
            final X509Certificate certificate = X509CertUtils.parseWithException(pemCertificate);
            final X500Principal principal = certificate.getSubjectX500Principal();
            final X500Name x500Name = new X500Name(principal.getName());
            final RDN x500NameRDN = x500Name.getRDNs(BCStyle.CN)[0];
            return IETFUtils.valueToString(x500NameRDN.getFirst().getValue());
        } catch (CertificateException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * Removes BEGIN, END markers, and all linebreaks from a PEM-String
     *
     * @param clientCertificate to be cleaned
     * @return cleaned Base64 encoded part of the certificate
     */
    public static String removeMarker(String clientCertificate) {
        return clientCertificate
                .replaceAll(PEM_BEGIN_MARKER, "")
                .replaceAll(PEM_END_MARKER, "")
                .replaceAll("\n", "")
                .replaceAll("\r", "");
    }
}
