package dev.fitko.fitconnect.core.crypto.constants;

import com.nimbusds.jose.EncryptionMethod;
import com.nimbusds.jose.JWEAlgorithm;

/**
 * Constants used for cryptographic operations in the FIT-Connect SDK. These constants define the
 * algorithms and methods used for encryption, decryption, and hashing.
 *
 * <p>For more information about the cryptographic requirements in FIT-Connect, see: <a
 * href="https://docs.fitko.de/fit-connect/docs/getting-started/get-started#verschlüsselte-übertragung-von-antragsdaten">FIT-Connect
 * Documentation - Encrypted Data Transmission</a>
 */
public final class CryptoConstants {

    private CryptoConstants() {
        // Utility class, prevent instantiation
    }

    /**
     * The default JWE (JSON Web Encryption) algorithm used for key wrapping. RSA-OAEP-256 provides
     * secure key wrapping using RSA with OAEP padding and SHA-256.
     */
    public static final JWEAlgorithm DEFAULT_JWE_ALGORITHM = JWEAlgorithm.RSA_OAEP_256;

    /**
     * The default JWE encryption method used for content encryption. A256GCM provides authenticated
     * encryption using AES-256 in GCM mode.
     */
    public static final EncryptionMethod DEFAULT_JWE_ENCRYPTION_METHOD = EncryptionMethod.A256GCM;

    /**
     * The default hash algorithm used for creating message digests. SHA-512 provides a 512-bit
     * (64-byte) hash value.
     */
    public static final String DEFAULT_HASH_ALGORITHM = "SHA-512";

    /**
     * The default HMAC algorithm used for message authentication. HmacSHA512 provides message
     * authentication using SHA-512.
     */
    public static final String DEFAULT_HMAC_ALGORITHM = "HmacSHA512";

    /**
     * The minimum key size in bits required for RSA keys. This ensures compliance with security
     * requirements.
     */
    public static final int MIN_RSA_KEY_SIZE = 2048;

    /**
     * The default key size in bits used for RSA key generation. This provides a good balance between
     * security and performance.
     */
    public static final int DEFAULT_RSA_KEY_SIZE = 2048;

    public static final String HASH_OF_ZERO_BYTES =
            "cf83e1357eefb8bdf1542850d66d8007d620e4050b5715dc83f4a921d36ce9ce47d0d13c5d85f2b0ff8318d2877eec2f63b931bd47417a81a538327af927da3e";
}
