package dev.fitko.fitconnect.core.cases;

import static dev.fitko.fitconnect.core.http.MimeTypes.APPLICATION_JOSE;
import static dev.fitko.fitconnect.core.http.MimeTypes.APPLICATION_JSON;
import static dev.fitko.fitconnect.core.utils.EventLogUtil.eventSubjectEqualsId;
import static dev.fitko.fitconnect.core.utils.EventLogUtil.eventToLogEntry;
import static dev.fitko.fitconnect.core.utils.EventLogUtil.getAuthTags;
import static dev.fitko.fitconnect.core.utils.EventLogUtil.getFilteredJwtsFromEventLog;
import static dev.fitko.fitconnect.core.utils.EventLogUtil.getJWTSFromEvents;
import static dev.fitko.fitconnect.core.utils.EventLogUtil.mapEventLogToEntries;

import com.nimbusds.jwt.SignedJWT;
import dev.fitko.fitconnect.api.domain.model.cases.Case;
import dev.fitko.fitconnect.api.domain.model.cases.Cases;
import dev.fitko.fitconnect.api.domain.model.event.Event;
import dev.fitko.fitconnect.api.domain.model.event.EventLog;
import dev.fitko.fitconnect.api.domain.model.event.EventLogEntry;
import dev.fitko.fitconnect.api.domain.model.event.Status;
import dev.fitko.fitconnect.api.domain.model.event.authtags.AuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.event.authtags.ValidatedAuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.event.problems.submission.InvalidEventLog;
import dev.fitko.fitconnect.api.domain.model.event.problems.submission.MissingAuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.event.problems.submission.NotExactlyOneSubmitEvent;
import dev.fitko.fitconnect.api.domain.model.reply.Reply;
import dev.fitko.fitconnect.api.domain.model.reply.SentReply;
import dev.fitko.fitconnect.api.domain.model.submission.SentSubmission;
import dev.fitko.fitconnect.api.domain.model.submission.Submission;
import dev.fitko.fitconnect.api.domain.validation.ValidationContext;
import dev.fitko.fitconnect.api.domain.validation.ValidationResult;
import dev.fitko.fitconnect.api.exceptions.internal.EventLogException;
import dev.fitko.fitconnect.api.exceptions.internal.RestApiException;
import dev.fitko.fitconnect.api.exceptions.internal.SubmitEventNotFoundException;
import dev.fitko.fitconnect.api.services.auth.OAuthService;
import dev.fitko.fitconnect.api.services.events.CaseService;
import dev.fitko.fitconnect.api.services.events.EventLogVerificationService;
import dev.fitko.fitconnect.api.services.http.HttpClient;
import dev.fitko.fitconnect.core.http.HttpHeaders;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CaseApiService implements CaseService {

    public static final String CASES_PATH = "/v2/cases";
    public static final String CASE_PATH = "/v2/cases/%s";
    public static final String CASES_EVENTS_PATH = "/v2/cases/%s/events";
    public static final String SUBMISSIONS_EVENTS_PATH = "/v2/submissions/%s/events";
    public static final String REPLIES_EVENTS_PATH = "/v2/replies/%s/events";
    private static final int PAGE_LIMIT = 100;

    private static final Logger LOGGER = LoggerFactory.getLogger(CaseApiService.class);

    private final OAuthService authService;
    private final HttpClient httpClient;
    private final EventLogVerificationService eventLogVerifier;
    private final String baseUrl;

    public CaseApiService(
            final OAuthService authService,
            final HttpClient httpClient,
            final EventLogVerificationService eventLogVerifier,
            final String baseUrl) {
        this.authService = authService;
        this.httpClient = httpClient;
        this.eventLogVerifier = eventLogVerifier;

        this.baseUrl = baseUrl;
    }

    @Override
    public List<EventLogEntry> getEventLogForCase(
            UUID destinationId, UUID caseId, AuthenticationTags authenticationTags) {
        final String caseEventsUrl = String.format(baseUrl + CASES_EVENTS_PATH, caseId);
        final List<String> eventLog = loadEventLog(caseEventsUrl);
        final List<SignedJWT> events = getJWTSFromEvents(eventLog);
        final SentSubmission sentSubmission = new SentSubmission(destinationId, null, caseId, authenticationTags);
        final ValidationContext ctx = getValidationContext(sentSubmission);
        checkForValidationErrors(eventLogVerifier.validateEventLogs(ctx, events));
        return mapEventLogToEntries(events);
    }

    @Override
    public List<EventLogEntry> getEventLogForSubmission(
            UUID destinationId, UUID caseId, UUID submissionId, AuthenticationTags authenticationTags) {
        final List<String> eventLog = loadEventLog(String.format(baseUrl + SUBMISSIONS_EVENTS_PATH, submissionId));
        final List<SignedJWT> events = getJWTSFromEvents(eventLog);
        final SentSubmission sentSubmission =
                new SentSubmission(destinationId, submissionId, caseId, authenticationTags);
        final ValidationContext ctx = getValidationContext(sentSubmission);
        checkForValidationErrors(eventLogVerifier.validateEventLogs(ctx, events));
        return mapEventLogToEntries(events);
    }

    @Override
    public ValidatedAuthenticationTags getAuthenticationTags(final Submission submission)
            throws RestApiException, EventLogException {
        final List<SignedJWT> submitEvents =
                getEvents(SUBMISSIONS_EVENTS_PATH, submission.getSubmissionId(), Event.SUBMIT_SUBMISSION);
        return getValidatedAuthenticationTags(submission.getDestinationId(), submission.getCaseId(), submitEvents);
    }

    @Override
    public ValidatedAuthenticationTags getAuthenticationTags(final Reply reply)
            throws RestApiException, EventLogException {
        final List<SignedJWT> submitEvents = getEvents(REPLIES_EVENTS_PATH, reply.getReplyId(), Event.SUBMIT_REPLY);
        return getValidatedAuthenticationTags(null, reply.getCaseId(), submitEvents);
    }

    @Override
    public void sendEvent(final UUID caseId, final String signedAndSerializedSET) {
        final String url = String.format(baseUrl + CASES_EVENTS_PATH, caseId);
        try {
            httpClient.post(url, buildHeaders(APPLICATION_JOSE), signedAndSerializedSET, Void.class);
        } catch (final RestApiException e) {
            throw new RestApiException("Sending event failed", e);
        }
    }

    @Override
    public Status getStatus(final SentSubmission sentSubmission) {
        final SignedJWT latestEvent = getLogFilteredById(SUBMISSIONS_EVENTS_PATH, sentSubmission.getSubmissionId());
        final ValidationContext validationContext = getValidationContext(sentSubmission);
        return getVerifiedStatus(validationContext, latestEvent);
    }

    @Override
    public Status getStatus(final SentReply reply) {
        final SignedJWT latestEvent = getLogFilteredById(REPLIES_EVENTS_PATH, reply.getReplyId());
        final ValidationContext contextWithoutAuthTagValidation =
                ValidationContext.withoutAuthTagValidation(reply.getCaseId());
        return getVerifiedStatus(contextWithoutAuthTagValidation, latestEvent);
    }

    @Override
    public Status getSubmissionSubmitState(final UUID caseId, final UUID submissionId) {
        final List<SignedJWT> submitEvents = getEvents(SUBMISSIONS_EVENTS_PATH, submissionId, Event.SUBMIT_SUBMISSION);
        if (submitEvents.size() != 1) {
            throw new SubmitEventNotFoundException("Event log does not contain exactly one submit event");
        }
        final var ctx = ValidationContext.withoutAuthTagValidation(caseId);
        checkForValidationErrors(eventLogVerifier.validateEventLogs(ctx, submitEvents));
        final EventLogEntry eventLogEntry = eventToLogEntry(submitEvents.get(0));
        return Status.fromEventLogEntry(eventLogEntry);
    }

    @Override
    public Cases listCases(final int limit, final int offset) {
        return loadCases(limit, offset);
    }

    @Override
    public Case getCase(final UUID caseId) {
        final String url = String.format(baseUrl + CASE_PATH, caseId);
        try {
            return httpClient.get(url, buildHeaders(), Case.class).getBody();
        } catch (final RestApiException e) {
            throw new RestApiException("Case query failed", e);
        }
    }

    private List<SignedJWT> getEvents(final String urlBasePath, final UUID submissionOrReplyId, final Event event) {
        final String caseEventsUrl = String.format(baseUrl + urlBasePath, submissionOrReplyId);
        final List<String> eventLog = loadEventLog(caseEventsUrl);
        return getFilteredJwtsFromEventLog(submissionOrReplyId, event, eventLog);
    }

    private List<String> loadEventLog(final String requestUrl) {
        final List<String> eventLogs = new ArrayList<>();
        int offset = 0;

        // we retrieve all available content, no matter how many pages exist.
        while (true) {
            final String url = requestUrl + "?limit=" + PAGE_LIMIT + "&offset=" + offset;
            final EventLog page = fetchPage(url, EventLog.class, "EventLog query failed");
            eventLogs.addAll(page.getEventLogs());

            if (isLastPage(page, offset, PAGE_LIMIT)) {
                break;
            }
            offset += PAGE_LIMIT;
        }

        return eventLogs;
    }

    private Cases loadCases(final int limit, final int offset) {
        final String url = baseUrl + CASES_PATH + "?limit=" + limit + "&offset=" + offset;
        return fetchPage(url, Cases.class, "Cases query failed");
    }

    private boolean isLastPage(final EventLog page, final int offset, final int limit) {
        final int currentFetchSize = page.getEventLogs().size();
        final int nextOffset = offset + limit;
        // this stop condition looks overengineered, because it is supposed to be robust towards race conditions
        // making "totalCount" and the "eventLogs" partial list inconsistent.
        // this implementation will always terminate and will never perform unnecessary requests.
        return currentFetchSize < limit || nextOffset >= page.getTotalCount();
    }

    private <T> T fetchPage(final String url, final Class<T> responseType, final String errorMessage) {
        try {
            return httpClient.get(url, buildHeaders(), responseType).getBody();
        } catch (final RestApiException e) {
            throw new RestApiException(errorMessage, e);
        }
    }

    private Status getVerifiedStatus(final ValidationContext validationContext, final SignedJWT latestEvent) {
        if (latestEvent == null) {
            LOGGER.info("No events found");
            return new Status();
        }
        checkForValidationErrors(eventLogVerifier.validateEventLogs(validationContext, List.of(latestEvent)));
        final EventLogEntry eventLogEntry = eventToLogEntry(latestEvent);
        return Status.fromEventLogEntry(eventLogEntry);
    }

    private SignedJWT getLogFilteredById(final String urlBasePath, final UUID id) {
        final String caseEventsUrl = String.format(baseUrl + urlBasePath, id);
        final List<String> eventLog = loadEventLog(caseEventsUrl);
        return getJWTSFromEvents(eventLog).stream()
                .filter(eventSubjectEqualsId(id))
                .reduce((first, second) -> second)
                .orElse(null);
    }

    private void checkForValidationErrors(final List<ValidationResult> validationResults) {
        if (!validationResults.isEmpty()) {
            throw new EventLogException(joinMessages(validationResults));
        }
    }

    private String joinMessages(final List<ValidationResult> validationResults) {
        return validationResults.stream()
                .map(ValidationResult::getError)
                .map(Exception::getMessage)
                .collect(Collectors.joining("\n"));
    }

    private Map<String, String> buildHeaders(final String contentType) {
        return new HashMap<>(Map.of(
                HttpHeaders.AUTHORIZATION,
                "Bearer " + authService.getCurrentToken().getAccessToken(),
                HttpHeaders.CONTENT_TYPE,
                contentType));
    }

    private ValidationContext getValidationContext(SentSubmission sentSubmission) {
        final UUID destinationId = sentSubmission.getDestinationId();
        final UUID caseId = sentSubmission.getCaseId();
        if (sentSubmission.getAuthenticationTags() == null) {
            return ValidationContext.withoutAuthTagValidation(destinationId, caseId);
        }
        return ValidationContext.withAuthTagValidation(destinationId, caseId, sentSubmission.getAuthenticationTags());
    }

    private ValidatedAuthenticationTags getValidatedAuthenticationTags(
            UUID destinationId, UUID caseId, List<SignedJWT> submitEvents) {

        final int submitEventSize = submitEvents.size();
        // https://docs.fitko.de/fit-connect/docs/receiving/verification/#genau-ein-submit-event
        if (submitEventSize != 1) {
            if (submitEventSize == 0) {
                LOGGER.warn("No submit event found in event-log, expected exactly 1");
            } else {
                LOGGER.warn("Eventlog contains {} submit events, expected exactly 1", submitEventSize);
            }
            return ValidatedAuthenticationTags.fromInvalidTags(new NotExactlyOneSubmitEvent());
        }

        final AuthenticationTags authenticationTags = getAuthTags(submitEvents.get(0));

        // https://docs.fitko.de/fit-connect/docs/receiving/verification/#authentication-tags-im-submit-event
        if (authenticationTags == null
                || authenticationTags.getMetadata() == null
                || authenticationTags.getData() == null) {
            return ValidatedAuthenticationTags.fromInvalidTags(new MissingAuthenticationTags());
        }

        final var ctx = ValidationContext.withAuthTagValidation(destinationId, caseId, authenticationTags);
        final List<ValidationResult> validationResults = eventLogVerifier.validateEventLogs(ctx, submitEvents);

        // https://docs.fitko.de/fit-connect/docs/receiving/verification/#struktur--und-signaturpr%C3%BCfung-der-security-event-tokens
        if (!validationResults.isEmpty()) {
            return ValidatedAuthenticationTags.fromInvalidTags(new InvalidEventLog(), joinMessages(validationResults));
        }

        return ValidatedAuthenticationTags.fromValidTags(authenticationTags);
    }

    private Map<String, String> buildHeaders() {
        return new HashMap<>(Map.of(
                HttpHeaders.AUTHORIZATION,
                "Bearer " + authService.getCurrentToken().getAccessToken(),
                HttpHeaders.ACCEPT,
                APPLICATION_JSON,
                HttpHeaders.ACCEPT_CHARSET,
                StandardCharsets.UTF_8.toString()));
    }
}
