# FIT-Connect Core Package

The core package provides essential utilities and services that form the foundation of the FIT-Connect SDK. This package contains low-level implementations and shared functionality used across the SDK.

## Main Components

### 1. Crypto (`core.crypto`)
- **JWECryptoService**: Handles encryption and decryption using JWE (JSON Web Encryption)
- **HashService**: Provides message digest and HMAC functionality
- **KeyGenerator**: Utility for generating RSA key pairs and JWKs
- **CertUtils**: Utilities for handling X.509 certificates

### 2. HTTP (`core.http`)
- **DefaultHttpClient**: Implementation of the HTTP client interface
- **HttpHeaders**: Common HTTP header constants
- **RetryInterceptor**: Handles request retries with exponential backoff
- **UserAgentInterceptor**: Manages User-Agent headers
- **ApiRequestInterceptor**: Handles API-specific request modifications

### 3. IO (`core.io`)
- **FileChunker**: Utility for splitting and merging large files into manageable chunks
- Supports streaming operations and file size management

### 4. Utils (`core.utils`)
- **Preconditions**: Argument validation utilities
- **Formatter**: Formatting utilities for time and file sizes
- **Strings**: String manipulation utilities
- **StopWatch**: Performance measurement utility
- **Version**: Semantic version handling

### 5. Schema (`core.schema`)
- **SchemaResourceProvider**: Manages schema resources and validation
- Supports both local and remote schema loading

### 6. Validation (`core.validation`)
- **DefaultValidationService**: Core validation service implementation
- **XmlSchemaValidator**: XML schema validation utilities

### 7. ZBP (`core.zbp`)
- **TokenGenerator**: Generates JWT tokens for ZBP authentication
- **ZBPApiService**: Core ZBP API service implementation

## Key Features

1. **Cryptographic Operations**
   - RSA encryption/decryption
   - JWE token handling
   - Hash generation and verification
   - Certificate management

2. **HTTP Communication**
   - Configurable timeouts
   - Retry mechanisms
   - Proxy support
   - SSL/TLS configuration

3. **File Handling**
   - Large file chunking
   - Stream processing
   - File size management

4. **Validation**
   - Schema validation
   - XML validation
   - Input validation

## Usage Examples

### Cryptographic Operations
```java
// Generate RSA key pair
KeyPair keyPair = KeyGenerator.buildRSAKeyPair(2048);

// Create hash
byte[] hash = hashService.createHash(data);

// Encrypt data
String encrypted = jweCryptoService.encrypt(data, publicKey);
```

### HTTP Client
```java
HttpClient client = new DefaultHttpClient(config);
HttpResponse response = client.get("https://api.example.com/data");
```

### File Chunking
```java
FileChunker chunker = new FileChunker();
List<File> chunks = chunker.chunkFile(filePath, chunkSize, outputDir);
```

## Dependencies

The core package depends on:
- `com.nimbusds.jose`: JWT and JWE handling
- `okhttp3`: HTTP client implementation
- `org.bouncycastle`: Cryptographic operations
- `com.fasterxml.jackson`: JSON processing
