package dev.fitko.fitconnect.core;

import com.nimbusds.jose.JWEObject;
import com.nimbusds.jose.jwk.RSAKey;
import com.nimbusds.jwt.SignedJWT;
import dev.fitko.fitconnect.api.FitConnectService;
import dev.fitko.fitconnect.api.domain.limits.submission.SubmissionLimits;
import dev.fitko.fitconnect.api.domain.model.cases.Case;
import dev.fitko.fitconnect.api.domain.model.cases.Cases;
import dev.fitko.fitconnect.api.domain.model.destination.Destination;
import dev.fitko.fitconnect.api.domain.model.destination.PublicDestination;
import dev.fitko.fitconnect.api.domain.model.event.Event;
import dev.fitko.fitconnect.api.domain.model.event.EventLogEntry;
import dev.fitko.fitconnect.api.domain.model.event.EventPayload;
import dev.fitko.fitconnect.api.domain.model.event.Status;
import dev.fitko.fitconnect.api.domain.model.event.authtags.AuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.event.authtags.ValidatedAuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.event.problems.Problem;
import dev.fitko.fitconnect.api.domain.model.metadata.Metadata;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.AttachmentForValidation;
import dev.fitko.fitconnect.api.domain.model.reply.AcceptReply;
import dev.fitko.fitconnect.api.domain.model.reply.AnnounceReply;
import dev.fitko.fitconnect.api.domain.model.reply.CreatedReply;
import dev.fitko.fitconnect.api.domain.model.reply.RepliesForPickup;
import dev.fitko.fitconnect.api.domain.model.reply.Reply;
import dev.fitko.fitconnect.api.domain.model.reply.SentReply;
import dev.fitko.fitconnect.api.domain.model.reply.SubmitReply;
import dev.fitko.fitconnect.api.domain.model.submission.AnnounceSubmission;
import dev.fitko.fitconnect.api.domain.model.submission.CreatedSubmission;
import dev.fitko.fitconnect.api.domain.model.submission.SentSubmission;
import dev.fitko.fitconnect.api.domain.model.submission.Submission;
import dev.fitko.fitconnect.api.domain.model.submission.SubmissionsForPickup;
import dev.fitko.fitconnect.api.domain.model.submission.SubmitSubmission;
import dev.fitko.fitconnect.api.domain.validation.ValidationResult;
import dev.fitko.fitconnect.api.domain.validation.VirusScanResult;
import dev.fitko.fitconnect.api.exceptions.internal.DecryptionException;
import dev.fitko.fitconnect.api.exceptions.internal.EncryptionException;
import dev.fitko.fitconnect.api.exceptions.internal.EventCreationException;
import dev.fitko.fitconnect.api.exceptions.internal.EventLogException;
import dev.fitko.fitconnect.api.exceptions.internal.InvalidKeyException;
import dev.fitko.fitconnect.api.exceptions.internal.RestApiException;
import dev.fitko.fitconnect.api.exceptions.internal.VirusScanException;
import dev.fitko.fitconnect.api.services.crypto.CryptoService;
import dev.fitko.fitconnect.api.services.destination.DestinationService;
import dev.fitko.fitconnect.api.services.events.CaseService;
import dev.fitko.fitconnect.api.services.events.SecurityEventService;
import dev.fitko.fitconnect.api.services.keys.KeyService;
import dev.fitko.fitconnect.api.services.reply.ReplyService;
import dev.fitko.fitconnect.api.services.submission.SubmissionService;
import dev.fitko.fitconnect.api.services.validation.ValidationService;
import dev.fitko.fitconnect.api.services.validation.VirusScanService;
import dev.fitko.fitconnect.client.attachments.ConcurrencyLimiter;
import java.io.InputStream;
import java.net.URI;
import java.nio.file.Path;
import java.util.List;
import java.util.UUID;
import lombok.AllArgsConstructor;
import lombok.Builder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** Default implementation of the FitConnectService interface. */
@Builder
@AllArgsConstructor
public class FitConnectDefaultService implements FitConnectService {

    private static final Logger LOGGER = LoggerFactory.getLogger(FitConnectDefaultService.class);

    private final ValidationService validationService;
    private final ReplyService replyService;
    private final CryptoService cryptoService;
    private final SubmissionService submissionService;
    private final DestinationService destinationService;
    private final CaseService caseService;
    private final KeyService keyService;
    private final SecurityEventService securityEventService;
    private final ConcurrencyLimiter concurrencyLimiter;
    private final VirusScanService virusScanService;

    @Override
    public ValidationResult validateMetadataSchema(final Metadata metadata) {
        LOGGER.info("Validating metadata");
        return validationService.validateMetadataSchema(metadata);
    }

    @Override
    public ValidationResult validateSubmissionMetadata(
            final Metadata metadata, final Submission submission, final AuthenticationTags authenticationTags)
            throws RestApiException {
        LOGGER.info("Validating submission metadata");
        final PublicDestination destination = submissionService.getDestination(submission.getDestinationId());
        return validationService.validateSubmissionMetadata(metadata, submission, destination, authenticationTags);
    }

    @Override
    public ValidationResult validateReplyMetadata(
            final Metadata metadata, final Reply reply, final AuthenticationTags authenticationTags)
            throws RestApiException {
        LOGGER.info("Validating reply metadata");
        final Case submissionCase = getCase(reply.getCaseId());
        final PublicDestination destination = submissionService.getDestination(submissionCase.getDestinationId());
        return validationService.validateReplyMetadata(metadata, reply, destination, authenticationTags);
    }

    @Override
    public ValidationResult validateData(
            final byte[] data, final String encryptedData, final Metadata metadata, final String authenticationTags) {
        LOGGER.info("Validating data");
        return validationService.validateData(data, encryptedData, metadata, authenticationTags);
    }

    @Override
    public ValidationResult validateAttachments(
            final List<AttachmentForValidation> attachmentsForValidation, final AuthenticationTags authenticationTags) {
        LOGGER.info("Validating attachments");
        return validationService.validateAttachments(attachmentsForValidation, authenticationTags);
    }

    @Override
    public ValidationResult validateJsonSchema(final byte[] json, final URI schemaUri) {
        LOGGER.info("Validating submission data against JSON schema");
        return validationService.validateJsonSubmissionDataSchema(json, schemaUri);
    }

    @Override
    public ValidationResult validateXmlSchema(final byte[] json, final URI schemaUri) {
        LOGGER.info("Validating submission data against XML schema");
        return validationService.validateXmlSubmissionDataSchema(json, schemaUri);
    }

    @Override
    public ValidationResult validateJsonFormat(final byte[] json) {
        LOGGER.info("Validating data JSON format");
        return validationService.validateJsonFormat(json);
    }

    @Override
    public ValidationResult validateXmlFormat(final byte[] xml) {
        LOGGER.info("Validating data XML format");
        return validationService.validateXmlFormat(xml);
    }

    @Override
    public ValidationResult validateCallback(
            final String hmac, final Long timestampInSeconds, final String httpBody, final String callbackSecret) {
        LOGGER.info("Validating callback integrity");
        return validationService.validateCallback(hmac, timestampInSeconds, httpBody, callbackSecret);
    }

    @Override
    public String encryptBytes(final RSAKey publicKey, final byte[] data, final String contentType)
            throws EncryptionException {
        return cryptoService.encryptBytes(publicKey, data, contentType);
    }

    @Override
    public JWEObject encryptInputStream(final RSAKey publicKey, final InputStream inputStream, final String contentType)
            throws EncryptionException {
        return cryptoService.encryptInputStream(publicKey, inputStream, contentType);
    }

    @Override
    public String encryptObject(final RSAKey encryptionKey, final Object obj, final String contentType)
            throws EncryptionException {
        return cryptoService.encryptObject(encryptionKey, obj, contentType);
    }

    @Override
    public byte[] decryptString(final RSAKey privateKey, final String encryptedContent) throws DecryptionException {
        return cryptoService.decryptToBytes(privateKey, encryptedContent);
    }

    @Override
    public String createHash(final byte[] data) {
        return cryptoService.hashBytes(data);
    }

    @Override
    public String createHash(final InputStream inputStream) {
        return cryptoService.hashStream(inputStream);
    }

    @Override
    public SubmissionsForPickup getAvailableSubmissions(final UUID destinationId, final int offset, final int limit)
            throws RestApiException {
        LOGGER.info("Loading available submissions {}-{} for destination {}", offset, offset + limit, destinationId);
        return submissionService.pollAvailableSubmissionsForDestination(destinationId, offset, limit);
    }

    @Override
    public CreatedSubmission announceSubmission(final AnnounceSubmission submission) throws RestApiException {
        LOGGER.info("Announcing new submission for destination {}", submission.getDestinationId());
        return submissionService.announceSubmission(submission);
    }

    @Override
    public Submission sendSubmission(final SubmitSubmission submission) throws RestApiException {
        LOGGER.info("Sending submission {}", submission.getSubmissionId());
        return submissionService.sendSubmission(submission);
    }

    @Override
    public void uploadSubmissionAttachment(
            final UUID submissionId, final UUID attachmentId, final String encryptedAttachment)
            throws RestApiException {
        try (final ConcurrencyLimiter.AcquiredSemaphore ignored = concurrencyLimiter.acquire()) {
            LOGGER.info("Uploading attachment {} for submission {}", attachmentId, submissionId);
            submissionService.uploadAttachment(submissionId, attachmentId, encryptedAttachment);
        }
    }

    @Override
    public void uploadSubmissionAttachmentStream(
            final UUID submissionId, final UUID attachmentId, final InputStream encryptedAttachment)
            throws RestApiException {
        try (final ConcurrencyLimiter.AcquiredSemaphore ignored = concurrencyLimiter.acquire()) {
            LOGGER.info("Uploading attachment {} for submission {}", attachmentId, submissionId);
            submissionService.uploadAttachmentStream(submissionId, attachmentId, encryptedAttachment);
        }
    }

    @Override
    public String getSubmissionAttachment(final UUID submissionId, final UUID attachmentId) throws RestApiException {
        try (final ConcurrencyLimiter.AcquiredSemaphore ignored = concurrencyLimiter.acquire()) {
            LOGGER.info("Loading attachment {} for submission {}", attachmentId, submissionId);
            return submissionService.getAttachment(submissionId, attachmentId);
        }
    }

    @Override
    public SubmissionLimits getDestinationAttachmentLimits(UUID destinationId) throws RestApiException {
        LOGGER.info("Loading attachment limits for destination {}", destinationId);
        return submissionService.getDestinationAttachmentLimits(destinationId);
    }

    @Override
    public SubmissionLimits getCaseAttachmentLimits(UUID caseId) throws RestApiException {
        LOGGER.info("Loading attachment limits for case {}", caseId);
        return submissionService.getCaseAttachmentLimits(caseId);
    }

    @Override
    public Submission getSubmission(final UUID submissionId) throws RestApiException {
        LOGGER.info("Loading submission {}", submissionId);
        return submissionService.getSubmission(submissionId);
    }

    @Override
    public void acceptSubmission(final EventPayload eventPayload) throws RestApiException, EventCreationException {
        LOGGER.info("Accepting submission {}", eventPayload.getSubmissionId());
        final SignedJWT confirmedSubmissionEvent = securityEventService.createAcceptSubmissionEvent(eventPayload);
        caseService.sendEvent(eventPayload.getCaseId(), confirmedSubmissionEvent.serialize());
        LOGGER.info("CONFIRMED submission {} successfully", eventPayload.getSubmissionId());
    }

    @Override
    public void rejectSubmission(final EventPayload eventPayload) throws RestApiException, EventCreationException {
        LOGGER.info("Rejecting submission {}", eventPayload.getSubmissionId());
        final SignedJWT rejectSubmissionEvent = securityEventService.createRejectSubmissionEvent(eventPayload);
        caseService.sendEvent(eventPayload.getCaseId(), rejectSubmissionEvent.serialize());
        LOGGER.info("REJECTED submission {}", eventPayload.getSubmissionId());
    }

    @Override
    public Destination getPrivateDestination(final UUID destinationId) throws RestApiException {
        LOGGER.info("Loading destination {}", destinationId);
        return destinationService.getDestination(destinationId);
    }

    @Override
    public PublicDestination getPublicDestination(final UUID destinationId) throws RestApiException {
        LOGGER.info("Loading destination {}", destinationId);
        return submissionService.getDestination(destinationId);
    }

    @Override
    public RSAKey getEncryptionKeyForDestination(final PublicDestination destination)
            throws RestApiException, InvalidKeyException {
        LOGGER.info("Loading encryption key for destination id {}", destination.getDestinationId());
        return keyService.getPublicEncryptionKey(destination);
    }

    @Override
    public RSAKey getEncryptionKeyForDestination(final UUID destinationId)
            throws RestApiException, InvalidKeyException {
        LOGGER.info("Loading encryption key for destination id {}", destinationId);
        final PublicDestination destination = getPublicDestination(destinationId);
        return keyService.getPublicEncryptionKey(destination);
    }

    @Override
    public List<EventLogEntry> getEventLogForCase(
            UUID destinationId, UUID caseId, AuthenticationTags authenticationTags) {
        LOGGER.info("Loading event log for case {}", caseId);
        return caseService.getEventLogForCase(destinationId, caseId, authenticationTags);
    }

    @Override
    public List<EventLogEntry> getEventLogForSubmission(
            UUID destinationId, UUID caseId, UUID submissionId, AuthenticationTags authenticationTags) {
        LOGGER.info("Loading event log for submission {}", submissionId);
        return caseService.getEventLogForSubmission(destinationId, caseId, submissionId, authenticationTags);
    }

    @Override
    public Status getStatus(final SentSubmission sentSubmission) throws RestApiException {
        LOGGER.info("Loading status of submission {}", sentSubmission.getSubmissionId());
        return caseService.getStatus(sentSubmission);
    }

    @Override
    public Status getStatus(final SentReply reply) throws RestApiException {
        LOGGER.info("Loading status of reply {}", reply.getReplyId());
        return caseService.getStatus(reply);
    }

    @Override
    public ValidatedAuthenticationTags getSubmissionAuthenticationTags(final Submission submission)
            throws RestApiException, EventLogException {
        LOGGER.info(
                "Loading authentication tags of {} event for submission {}",
                Event.SUBMIT_SUBMISSION,
                submission.getSubmissionId());
        return caseService.getAuthenticationTags(submission);
    }

    @Override
    public ValidatedAuthenticationTags getReplyAuthenticationTags(final Reply reply)
            throws RestApiException, EventLogException {
        LOGGER.info("Loading authentication tags of {} event for reply {}", Event.SUBMIT_REPLY, reply.getReplyId());
        return caseService.getAuthenticationTags(reply);
    }

    @Override
    public Cases listCases(final int limit, final int offset) throws RestApiException, EventLogException {
        LOGGER.info("Loading active cases");
        return caseService.listCases(limit, offset);
    }

    @Override
    public Case getCase(final UUID caseId) throws RestApiException, EventLogException {
        LOGGER.info("Loading case {}", caseId);
        return caseService.getCase(caseId);
    }

    @Override
    public Reply getReply(final UUID replyId) throws RestApiException {
        LOGGER.info("Loading reply {}", replyId);
        return replyService.getReply(replyId);
    }

    @Override
    public RepliesForPickup getAvailableReplies(final int limit, final int offset) throws RestApiException {
        LOGGER.info("Loading available replies");
        return replyService.getAvailableReplies(limit, offset);
    }

    @Override
    public SentReply submitReply(final UUID replyId, final SubmitReply submitReply) {
        LOGGER.info("Submitting reply {}", replyId);
        return replyService.submitReply(replyId, submitReply);
    }

    @Override
    public CreatedReply announceReply(final AnnounceReply announceReply) {
        LOGGER.info("Announcing new reply for case {}", announceReply.getCaseId());
        return replyService.announceReply(announceReply);
    }

    @Override
    public String acceptReply(final UUID replyId, final AcceptReply acceptReply) {
        LOGGER.info("Accepting reply {}", replyId);
        return replyService.acceptReply(replyId, acceptReply);
    }

    @Override
    public String rejectReply(final UUID replyId, final List<Problem> problems) {
        LOGGER.info("Rejecting reply {}", replyId);
        return replyService.rejectReply(replyId, problems);
    }

    @Override
    public Status getSubmitState(final Submission submission) {
        LOGGER.info("Loading submit state for submission {}", submission.getSubmissionId());
        return caseService.getSubmissionSubmitState(submission.getCaseId(), submission.getSubmissionId());
    }

    @Override
    public void uploadReplyAttachment(final UUID replyId, final UUID attachmentId, final String encryptedAttachment) {
        try (final ConcurrencyLimiter.AcquiredSemaphore ignored = concurrencyLimiter.acquire()) {
            LOGGER.info("Uploading attachment {} for reply {}", attachmentId, replyId);
            replyService.uploadAttachment(replyId, attachmentId, encryptedAttachment);
        }
    }

    @Override
    public String getReplyAttachment(final UUID replyId, final UUID attachmentId) {
        try (final ConcurrencyLimiter.AcquiredSemaphore ignored = concurrencyLimiter.acquire()) {
            LOGGER.info("Loading attachment {} for reply {}", attachmentId, replyId);
            return replyService.getAttachment(replyId, attachmentId);
        }
    }

    @Override
    public VirusScanResult scanBytesForViruses(byte[] data) throws VirusScanException {
        LOGGER.info("Scanning byte array for viruses");
        return virusScanService.scanBytes(data);
    }

    @Override
    public VirusScanResult scanStreamForViruses(InputStream inputStream) throws VirusScanException {
        LOGGER.info("Scanning input stream for viruses");
        return virusScanService.scanStream(inputStream);
    }

    @Override
    public VirusScanResult scanFileForViruses(Path filePath) throws VirusScanException {
        LOGGER.info("Scanning file for viruses");
        return virusScanService.scanFile(filePath);
    }
}
