package dev.fitko.fitconnect.client.zbp;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.jwk.RSAKey;
import dev.fitko.fitconnect.api.domain.zbp.AuthorKeyPair;
import dev.fitko.fitconnect.api.domain.zbp.ZBPEnvelope;
import dev.fitko.fitconnect.api.domain.zbp.message.CreateMessage;
import dev.fitko.fitconnect.api.domain.zbp.state.CreateState;
import dev.fitko.fitconnect.core.crypto.utils.CertUtils;
import dev.fitko.fitconnect.core.zbp.TokenGenerator;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.Signature;
import java.security.SignatureException;
import java.util.Base64;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;

/**
 * Wraps a {@link CreateMessage} or {@link CreateState} in an {@link ZBPEnvelope} that will be sent
 * to the ZBP via FIT-Connect.
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public final class ZBPEnvelopeBuilder {

    private static final ObjectMapper MAPPER = ObjectMapperProvider.createObjectMapper();

    /**
     * Write payload to a {@link ZBPEnvelope} as byte[] that is signed with the authors private key.
     *
     * @param payload the @{@link CreateMessage} or {@link CreateState} payload that is written to
     *     the {@link ZBPEnvelope}
     * @param authorKeyPair pair of author client certificate and private key the token and content
     *     signature is created from
     * @return signed {@link ZBPEnvelope} as byte array
     */
    public static byte[] fromAuthorPayload(Object payload, AuthorKeyPair authorKeyPair) {
        final String authorCertificate = authorKeyPair.getAuthorCertificateAsPem();
        final RSAKey authorPrivateKey = authorKeyPair.getAuthorPrivateKey();
        final String signer = CertUtils.getSubjectCNFromCertificate(authorCertificate);
        final String token = TokenGenerator.buildToken(authorPrivateKey, signer);
        final ZBPEnvelope envelope = buildEnvelope(payload, authorPrivateKey, authorCertificate, token);
        return serializeToByteArray(envelope);
    }

    /**
     * Write payload to a {@link ZBPEnvelope} that is signed with the senders private key.
     *
     * @param payload the @{@link CreateMessage} or {@link CreateState} payload that is written to
     *     the {@link ZBPEnvelope}
     * @param senderPrivateKey sender private key
     * @param authorCertificate client certificate of the author
     * @param authorToken token of the author
     * @return signed {@link ZBPEnvelope} as byte array
     */
    public static ZBPEnvelope fromSenderPayload(
            Object payload, RSAKey senderPrivateKey, String authorCertificate, String authorToken) {
        return buildEnvelope(payload, senderPrivateKey, authorCertificate, authorToken);
    }

    private static ZBPEnvelope buildEnvelope(
            Object payload, RSAKey privateKey, String authorCertificate, String authorToken) {
        return ZBPEnvelope.builder()
                .content(serializeToJson(payload))
                .sha512sum(signPayload(payload, privateKey))
                .authorCertificate(authorCertificate)
                .authorToken(authorToken)
                .build();
    }

    private static String signPayload(Object input, RSAKey key) {
        try {
            Signature signature = Signature.getInstance("SHA512withRSA");
            signature.initSign(key.toPrivateKey());
            signature.update(MAPPER.writeValueAsBytes(input));
            final byte[] encoded = Base64.getEncoder().encode(signature.sign());
            return new String(encoded, StandardCharsets.UTF_8);
        } catch (NoSuchAlgorithmException
                | SignatureException
                | InvalidKeyException
                | JOSEException
                | JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    private static byte[] serializeToByteArray(Object obj) {
        try {
            return MAPPER.writeValueAsBytes(obj);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    private static String serializeToJson(Object payload) {
        return new String(serializeToByteArray(payload), StandardCharsets.UTF_8);
    }
}
