package dev.fitko.fitconnect.client.zbp;

import dev.fitko.fitconnect.api.domain.model.attachment.Attachment;
import dev.fitko.fitconnect.api.domain.zbp.attachment.ZBPApiAttachment;
import dev.fitko.fitconnect.api.domain.zbp.attachment.ZBPAttachmentMetadata;
import dev.fitko.fitconnect.api.services.crypto.MessageDigestService;
import dev.fitko.fitconnect.core.crypto.HashService;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;

@NoArgsConstructor(access = AccessLevel.PRIVATE)
public class ZBPAttachmentMetadataBuilder {

    private static final MessageDigestService HASH_SERVICE = new HashService();

    /**
     * Create ZBP attachment metadata.
     *
     * @param filename name of the attachment file
     * @param data attachment content as byte[]
     * @see ZBPAttachmentMetadata#ZBPAttachmentMetadata(String, String, int)
     */
    public static ZBPAttachmentMetadata from(String filename, byte[] data) {
        return new ZBPAttachmentMetadata(filename, calcHash(data), data.length);
    }

    /**
     * Create ZBP attachment metadata from a {@link ZBPApiAttachment}.
     *
     * @param zbpApiAttachment attachment the metadata is created for
     * @see ZBPAttachmentMetadata#ZBPAttachmentMetadata(String, String, int)
     */
    public static ZBPAttachmentMetadata from(ZBPApiAttachment zbpApiAttachment) {
        return new ZBPAttachmentMetadata(
                zbpApiAttachment.getFilename(),
                calcHash(zbpApiAttachment.getData()),
                zbpApiAttachment.getData().length);
    }

    /**
     * Create ZBP attachment metadata from a FIT-Connect {@link Attachment}.
     *
     * @param fitConnectAttachment attachment the metadata is created for
     * @see ZBPAttachmentMetadata#ZBPAttachmentMetadata(String, String, int)
     */
    public static ZBPAttachmentMetadata from(Attachment fitConnectAttachment) {
        return new ZBPAttachmentMetadata(
                fitConnectAttachment.getFileName(),
                calcHash(fitConnectAttachment.getDataAsBytes()),
                fitConnectAttachment.getDataAsBytes().length);
    }

    private static String calcHash(byte[] data) {
        final byte[] hash = HASH_SERVICE.createHash(data);
        return HASH_SERVICE.toHexString(hash);
    }
}
