package dev.fitko.fitconnect.client.util;

import static dev.fitko.fitconnect.api.config.ApplicationConfig.LEIKA_KEY_PATTERN;
import static dev.fitko.fitconnect.api.config.ApplicationConfig.MIN_METADATA_VERSION_WITH_CHUNKING_SUPPORT;
import static dev.fitko.fitconnect.core.utils.Preconditions.checkArgumentAndThrow;

import dev.fitko.fitconnect.api.FitConnectService;
import dev.fitko.fitconnect.api.config.Version;
import dev.fitko.fitconnect.api.config.chunking.AttachmentChunkingConfig;
import dev.fitko.fitconnect.api.domain.model.attachment.Attachment;
import dev.fitko.fitconnect.api.domain.model.callback.Callback;
import dev.fitko.fitconnect.api.domain.model.destination.DestinationReplyChannels;
import dev.fitko.fitconnect.api.domain.model.destination.PublicDestination;
import dev.fitko.fitconnect.api.domain.model.destination.Service;
import dev.fitko.fitconnect.api.domain.model.destination.replychannels.DestinationFitConnect;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.Purpose;
import dev.fitko.fitconnect.api.domain.model.metadata.data.MimeType;
import dev.fitko.fitconnect.api.domain.model.metadata.data.SubmissionSchema;
import dev.fitko.fitconnect.api.domain.model.reply.replychannel.ReplyChannel;
import dev.fitko.fitconnect.api.domain.sender.SendableEncryptedSubmission;
import dev.fitko.fitconnect.api.domain.sender.SendableSubmission;
import dev.fitko.fitconnect.api.domain.subscriber.SendableReply;
import dev.fitko.fitconnect.api.domain.validation.ValidationResult;
import java.net.URI;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/** Validate submission and reply payload against the properties and services of a destination. */
public class SubmissionValidator {

    private final FitConnectService fitConnectService;

    public SubmissionValidator(final FitConnectService fitConnectService) {
        this.fitConnectService = fitConnectService;
    }

    /**
     * Checks if the encrypted submission payload is in a valid state for sending.
     *
     * @param sendableEncryptedSubmission payload to be checked
     * @throws IllegalArgumentException if one of the checks fails
     */
    public void ensureValidDataPayload(
            final SendableEncryptedSubmission sendableEncryptedSubmission, PublicDestination destination) {

        checkArgumentAndThrow(sendableEncryptedSubmission == null, "Encrypted payload must not be null.");
        checkArgumentAndThrow(
                sendableEncryptedSubmission.getData() == null, "Encrypted data is mandatory, but was null.");
        checkArgumentAndThrow(
                sendableEncryptedSubmission.getMetadata() == null, "Encrypted metadata must not be null.");

        testDefaults(
                destination,
                sendableEncryptedSubmission.getServiceType().getIdentifier(),
                sendableEncryptedSubmission.getCallback());
    }

    /**
     * Checks if the unencrypted submission payload is in a valid state for sending.
     *
     * @param sendableSubmission payload to be checked
     * @throws IllegalArgumentException if one of the checks fails
     */
    public void ensureValidDataPayload(final SendableSubmission sendableSubmission, PublicDestination destination) {

        checkArgumentAndThrow(sendableSubmission == null, "Submission must not be null.");

        testDataFormat(
                sendableSubmission.getAttachments(),
                sendableSubmission.getSubmissionSchema().getMimeType(),
                sendableSubmission.getData());
        testDefaults(sendableSubmission, destination);
    }

    private void testDataFormat(List<Attachment> attachments, MimeType dataMimeType, byte[] data) {
        final List<Attachment> dataAttachments = attachments.stream()
                .filter(a -> a.getPurpose().equals(Purpose.DATA))
                .collect(Collectors.toList());
        if (dataAttachments.size() > 1) {
            throw new IllegalArgumentException(
                    "Data as an attachment can only be set once, found " + dataAttachments.size() + " entries");
        } else if (dataAttachments.size() == 1) {
            testOnValidDataFormat(dataAttachments.get(0).getDataAsBytes(), dataMimeType);
        } else {
            checkArgumentAndThrow(
                    data == null || data.length == 0, "Data payload is mandatory, but was null or empty.");
            testOnValidDataFormat(data, dataMimeType);
        }
    }

    /**
     * Checks if the unencrypted reply payload is in a valid state for sending.
     *
     * @param sendableReply reply payload to be checked
     * @throws IllegalArgumentException if one of the checks fails
     */
    public void ensureValidDataPayload(final SendableReply sendableReply, PublicDestination destination) {

        checkArgumentAndThrow(sendableReply == null, "Reply must not be null.");
        checkArgumentAndThrow(
                sendableReply.getReplyEncryptionKey() == null,
                "Public reply encryption key is mandatory, but was null");

        checkMatchingDataSchemaOnDestination(
                sendableReply.getSchemaUri(), sendableReply.getDataMimeType(), destination);
        testDataFormat(sendableReply.getAttachments(), sendableReply.getDataMimeType(), sendableReply.getData());
    }

    /**
     * Tests if the given destination has support for attachment chunking.
     *
     * <p>Throws an exception if a destination doesn't support chunking and any of these preconditions
     * is true:
     *
     * <ul>
     *   <li>all attachments should be chunked
     *   <li>any large attachments exist that should be chunked automatically
     * </ul>
     *
     * @param destination the destination to be checked
     * @param attachments list of attachments that are checked for the presence of large attachments
     * @param chunkingConfig config object for attachment chunking
     * @return boolean if chucking is provided by the destination
     */
    public boolean destinationSupportsAttachmentChunking(
            PublicDestination destination, List<Attachment> attachments, AttachmentChunkingConfig chunkingConfig) {

        checkArgumentAndThrow(
                destination.getMetadataVersions() == null,
                "Metadata versions not present on destination " + destination.getDestinationId());

        boolean metadataSupportsChunking = destination.getMetadataVersions().stream()
                .map(Version::new)
                .anyMatch(version -> version.isGreaterOrEqualThan(MIN_METADATA_VERSION_WITH_CHUNKING_SUPPORT));

        if (!metadataSupportsChunking
                && !attachments.isEmpty()
                && (chunkingConfig.isChunkAllAttachments()
                        || attachments.stream().anyMatch(Attachment::isLargeAttachment))) {
            throw new IllegalArgumentException(
                    "Submission contains chunked attachments but destination does not support chunking. At least metadata version "
                            + MIN_METADATA_VERSION_WITH_CHUNKING_SUPPORT
                            + " is required.");
        }

        return metadataSupportsChunking;
    }

    private void testDefaults(
            final PublicDestination destination, final String serviceIdentifier, final Callback callback) {

        checkArgumentAndThrow(destination == null, "DestinationId is mandatory, but was null.");
        checkArgumentAndThrow(serviceIdentifier == null, "Leika key is mandatory, but was null.");
        checkArgumentAndThrow(
                invalidLeikaKeyPattern(serviceIdentifier),
                "LeikaKey has invalid format, please follow: ^urn:[a-z0-9][a-z0-9-]{0,31}:[a-z0-9()+,.:=@;$_!*'%/?#-]+$.");

        if (serviceTypeDoesNotMatchDestination(destination, serviceIdentifier)) {
            throw new IllegalArgumentException(
                    "Provided service type '" + serviceIdentifier + "' is not allowed by the destination ");
        }
        if (callback != null) {
            checkCallbackFormat(callback);
        }
    }

    private void testDefaults(final SendableSubmission sendableSubmission, PublicDestination destination) {

        checkArgumentAndThrow(
                sendableSubmission.getDestinationId() == null, "DestinationId is mandatory, but was null.");
        checkArgumentAndThrow(
                sendableSubmission.getServiceType().getIdentifier() == null, "Leika key is mandatory, but was null.");
        checkArgumentAndThrow(
                invalidLeikaKeyPattern(sendableSubmission.getServiceType().getIdentifier()),
                "LeikaKey has invalid format, please follow: ^urn:[a-z0-9][a-z0-9-]{0,31}:[a-z0-9()+,.:=@;$_!*'%/?#-]+$.\")");

        if (serviceTypeDoesNotMatchDestination(
                destination, sendableSubmission.getServiceType().getIdentifier())) {
            throw new IllegalArgumentException("Provided service type '"
                    + sendableSubmission.getServiceType().getIdentifier()
                    + "' is not allowed by the destination ");
        }
        final SubmissionSchema submissionSchema = sendableSubmission.getSubmissionSchema();
        checkMatchingDataSchemaOnDestination(
                submissionSchema.getSchemaUri(), submissionSchema.getMimeType(), destination);
        checkMatchingProcessingStandardsOnDestination(
                destination.getPublicServices(),
                sendableSubmission.getReplyChannel(),
                sendableSubmission.getServiceType().getIdentifier(),
                sendableSubmission.getSubmissionSchema());
        if (sendableSubmission.getCallback() != null) {
            checkCallbackFormat(sendableSubmission.getCallback());
        }
    }

    private void checkMatchingProcessingStandardsOnDestination(
            final Set<Service> services,
            ReplyChannel submissionReplyChannel,
            String serviceIdentifier,
            final SubmissionSchema submissionSchema) {
        if (processingStandardsDoNotMatchDestination(
                services, submissionReplyChannel, serviceIdentifier, submissionSchema)) {
            throw new IllegalArgumentException(
                    "FIT-Connect reply channel processing standard(s) do not match any of the destination services");
        }
    }

    private boolean processingStandardsDoNotMatchDestination(
            final Set<Service> services,
            ReplyChannel submissionReplyChannel,
            String serviceIdentifier,
            final SubmissionSchema submissionSchema) {
        if (services == null
                || services.isEmpty()
                || services.stream().noneMatch(s -> s.getReplyChannels() != null)
                || submissionReplyChannel == null
                || submissionReplyChannel.getFitConnect() == null) {
            return false;
        }

        final Optional<DestinationFitConnect> matchingStandardsFound = services.stream()
                .filter(service -> service.getIdentifier().equals(serviceIdentifier))
                .filter(service ->
                        service.getSubmissionSchemas().stream().anyMatch(schema -> schema.equals(submissionSchema)))
                .map(Service::getReplyChannels)
                .map(DestinationReplyChannels::getDestinationFitConnect)
                .filter(Objects::nonNull)
                .filter(matchingProcessingStandards(submissionReplyChannel))
                .findAny();

        return matchingStandardsFound.isEmpty();
    }

    private static Predicate<DestinationFitConnect> matchingProcessingStandards(ReplyChannel submissionReplyChannel) {
        return destinationFitConnect -> new HashSet<>(destinationFitConnect.getProcessStandards())
                .containsAll(submissionReplyChannel.getFitConnect().getProcessStandards());
    }

    private void checkMatchingDataSchemaOnDestination(
            final URI submissionSchemaUri, MimeType dataMimeType, final PublicDestination destination) {
        if (mimeTypeAndSchemaUriDoNotMatchDestination(destination, dataMimeType, submissionSchemaUri)) {
            final String allowedSchemas = destination.getPublicServices().stream()
                    .flatMap(service -> service.getSubmissionSchemas().stream())
                    .filter(Objects::nonNull)
                    .map(schema -> schema.getMimeType() + " => " + schema.getSchemaUri())
                    .collect(Collectors.joining("\n"));
            if (allowedSchemas.isEmpty()) {
                throw new IllegalArgumentException(
                        "Destination has no available schemaUri for requested uri " + submissionSchemaUri);
            }
            throw new IllegalArgumentException(
                    "Please specify a valid schemaUri for the submission. The destination allows: " + allowedSchemas);
        }
    }

    private boolean serviceTypeDoesNotMatchDestination(
            final PublicDestination destination, final String serviceIdentifier) {
        return destination.getPublicServices().stream()
                .map(Service::getIdentifier)
                .filter(Objects::nonNull)
                .filter(destinationServiceIdentifier -> destinationServiceIdentifier.equals(serviceIdentifier))
                .findFirst()
                .isEmpty();
    }

    private boolean mimeTypeAndSchemaUriDoNotMatchDestination(
            final PublicDestination destination, final MimeType mimeType, final URI schemaUri) {
        return destination.getPublicServices().stream()
                .flatMap(service -> service.getSubmissionSchemas().stream())
                .filter(Objects::nonNull)
                .filter(schema -> schema.getMimeType().equals(mimeType))
                .filter(schema -> schema.getSchemaUri().equals(schemaUri))
                .findFirst()
                .isEmpty();
    }

    private void testOnValidDataFormat(final byte[] data, MimeType dataMimeType) {
        if (dataMimeType.equals(MimeType.APPLICATION_JSON)) {
            checkJsonFormat(data);
        } else if (dataMimeType.equals(MimeType.APPLICATION_XML)) {
            checkXmlFormat(data);
        }
    }

    private void checkXmlFormat(final byte[] xmlData) {
        final ValidationResult validationResult = fitConnectService.validateXmlFormat(xmlData);
        if (validationResult.hasError()) {
            throw new IllegalArgumentException("Data is not in expected xml format, please provide valid xml: "
                    + validationResult.getError().getMessage());
        }
    }

    private void checkJsonFormat(final byte[] jsonData) {
        final ValidationResult validationResult = fitConnectService.validateJsonFormat(jsonData);
        if (validationResult.hasError()) {
            throw new IllegalArgumentException("Data is not in expected json format, please provide valid json: "
                    + validationResult.getError().getMessage());
        }
    }

    private void checkCallbackFormat(final Callback callback) {
        if (callback.getUri() == null || !callback.getUri().getScheme().equals("https")) {
            throw new IllegalArgumentException(
                    "Callback URI " + callback.getUri() + " must be a secure https connection");
        }
        if (callback.getSecret() == null || callback.getSecret().length() < 32) {
            throw new IllegalArgumentException("Callback secret must have a length of at least 32 characters");
        }
    }

    private boolean invalidLeikaKeyPattern(final String leikaKey) {
        return !LEIKA_KEY_PATTERN.matcher(leikaKey).matches();
    }
}
