package dev.fitko.fitconnect.client.util;

import dev.fitko.fitconnect.api.config.resources.SchemaResourceConfig;
import dev.fitko.fitconnect.api.domain.schema.SchemaResources;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectInitialisationException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.yaml.snakeyaml.Yaml;

@NoArgsConstructor(access = AccessLevel.PRIVATE)
public final class ResourceLoadingUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(ResourceLoadingUtils.class);

    public static <T> T loadYaml(String path, Class<T> returnType) {
        try (final InputStream is = ResourceLoadingUtils.class.getClassLoader().getResourceAsStream(path)) {
            return new Yaml().loadAs(is, returnType);
        } catch (final Exception e) {
            throw new FitConnectInitialisationException("Yaml resource from path  " + path + " could not be loaded", e);
        }
    }

    public static String readKeyFromPath(final String keyPath) {
        try {
            return Files.readString(Path.of(keyPath));
        } catch (final IOException | NullPointerException e) {
            LOGGER.error("Could not read key from path '{}'. Please provide a valid path in your config.yml", keyPath);
            throw new FitConnectInitialisationException(e.getMessage(), e);
        }
    }

    public static SchemaResources loadAllSchemaResources(
            String schemaConfigName, Map<String, String> submissionDataSchemas) {
        final var schemaResourceConfig = loadYaml(schemaConfigName, SchemaResourceConfig.class);
        final List<String> metadataSchemas =
                readSchemaResourcesFromPaths(schemaResourceConfig.getMetadataSchemaPaths());
        final List<String> setSchemas = readSchemaResourcesFromPaths(schemaResourceConfig.getSetSchemaPaths());
        final List<String> destinationSchemas =
                readSchemaResourcesFromPaths(List.of(schemaResourceConfig.getDestinationSchemaPath()));
        return new SchemaResources(setSchemas, metadataSchemas, destinationSchemas, submissionDataSchemas);
    }

    public static String readResourceToString(final String resourcePath) {
        return new String(readResourceToByteArray(resourcePath), StandardCharsets.UTF_8);
    }

    private static byte[] readResourceToByteArray(final String resourcePath) {
        try (final InputStream in = ResourceLoadingUtils.class.getResourceAsStream(resourcePath)) {
            return Objects.requireNonNull(in).readAllBytes();
        } catch (final IOException | NullPointerException exception) {
            throw new FitConnectInitialisationException("Failed loading resource " + resourcePath, exception);
        }
    }

    private static List<String> readSchemaResourcesFromPaths(List<String> schemaPaths) {
        return schemaPaths.stream()
                .map(ResourceLoadingUtils::readResourceToByteArray)
                .map(String::new)
                .collect(Collectors.toList());
    }
}
