package dev.fitko.fitconnect.client.util;

import static com.nimbusds.jose.JWEAlgorithm.RSA_OAEP_256;
import static com.nimbusds.jose.jwk.KeyOperation.UNWRAP_KEY;
import static com.nimbusds.jose.jwk.KeyOperation.WRAP_KEY;

import com.nimbusds.jose.jwk.JWK;
import dev.fitko.fitconnect.api.domain.crypto.JWKPair;
import dev.fitko.fitconnect.core.crypto.utils.KeyGenerator;
import java.security.KeyPair;
import java.util.UUID;

/**
 * JWK key generator that creates a {@link JWKPair} of public and private key used in reply channel
 * communication.
 *
 * @see dev.fitko.fitconnect.client.sender.ReplyReceiver
 * @see dev.fitko.fitconnect.client.subscriber.ReplySender
 */
public final class ReplyChannelKeyGenerator {

    private static final int DEFAULT_KEY_SIZE = 4096;
    private static final String KEY_ID_PREFIX = "reply-key:";

    private ReplyChannelKeyGenerator() {}

    /**
     * Generates a pair of public and private JWK to be used in reply channel communication. Those
     * JWKs are ephemeral keys and do not contain a V-PKI X509-cert.
     *
     * @return key pair of public and private key.
     */
    public static JWKPair generateKeyPair() {
        return generateKeyPair(UUID.randomUUID());
    }

    /**
     * Generates a pair of public and private JWK to be used in reply channel communication. Those
     * JWKs are ephemeral keys and do NOT contain a V-PKI X509-cert.
     *
     * @param caseId uses the reply caseId as part of the keys ids as in <b>"kid":
     *     "reply-key:{caseId}"</b>
     * @return key pair of public and private key.
     */
    public static JWKPair generateKeyPair(final UUID caseId) {

        final String keyId = KEY_ID_PREFIX + caseId;
        final KeyPair keyPair = KeyGenerator.buildRSAKeyPair(DEFAULT_KEY_SIZE);

        final JWK publicEncryptionKey = KeyGenerator.buildJWK(keyPair, keyId, WRAP_KEY, RSA_OAEP_256);
        final JWK privateDecryptionKey = KeyGenerator.buildJWK(keyPair, keyId, UNWRAP_KEY, RSA_OAEP_256);

        return new JWKPair(publicEncryptionKey.toPublicJWK(), privateDecryptionKey);
    }
}
