package dev.fitko.fitconnect.client.util;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import dev.fitko.fitconnect.api.domain.model.metadata.Metadata;
import dev.fitko.fitconnect.api.domain.model.metadata.v1.MetadataV1;
import dev.fitko.fitconnect.api.domain.model.metadata.v2.MetadataV2;
import java.io.IOException;

public final class MetadataDeserializationHelper {

    private MetadataDeserializationHelper() {}

    private static final ObjectMapper DEFAULT_MAPPER = new ObjectMapper();

    /**
     * Deserializes metadata bytes to the appropriate Metadata implementation based on the schema version.
     *
     * @param mapper the ObjectMapper to use for deserialization
     * @param metadataBytes the JSON metadata as byte array
     * @return MetadataV1 for schema versions 1.x, MetadataV2 for schema versions 2.x
     * @throws IOException if deserialization fails or unsupported schema version is detected
     */
    public static Metadata deserializeMetadata(ObjectMapper mapper, byte[] metadataBytes) throws IOException {
        ObjectNode root = (ObjectNode) mapper.readTree(metadataBytes);
        String schemaUri = root.has("$schema") ? root.get("$schema").asText() : "";

        if (schemaUri.contains("/2.")) {
            return mapper.treeToValue(root, MetadataV2.class);
        } else if (schemaUri.contains("/1.")) {
            return mapper.treeToValue(root, MetadataV1.class);
        } else {
            throw new IOException("Failed to deserialize metadata. Unknown metadata version: " + schemaUri);
        }
    }

    /**
     * Deserializes metadata bytes to the appropriate Metadata implementation based on the schema version.
     *
     * @param metadataBytes the JSON metadata as byte array
     * @return MetadataV1 for schema versions 1.x, MetadataV2 for schema versions 2.x
     * @throws IOException if deserialization fails or unsupported schema version is detected
     */
    public static Metadata deserializeMetadata(byte[] metadataBytes) throws IOException {
        return deserializeMetadata(DEFAULT_MAPPER, metadataBytes);
    }
}
