package dev.fitko.fitconnect.client.util;

import static dev.fitko.fitconnect.api.config.defaults.SchemaConfig.METADATA_SCHEMA_PATH;
import static java.util.stream.Collectors.toList;

import dev.fitko.fitconnect.api.config.Version;
import dev.fitko.fitconnect.api.domain.model.attachment.AttachmentPayload;
import dev.fitko.fitconnect.api.domain.model.destination.PublicDestination;
import dev.fitko.fitconnect.api.domain.model.metadata.ContentStructure;
import dev.fitko.fitconnect.api.domain.model.metadata.Hash;
import dev.fitko.fitconnect.api.domain.model.metadata.Metadata;
import dev.fitko.fitconnect.api.domain.model.metadata.SignatureType;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.ApiAttachment;
import dev.fitko.fitconnect.api.domain.model.metadata.data.Data;
import dev.fitko.fitconnect.api.domain.model.metadata.data.SubmissionSchema;
import dev.fitko.fitconnect.api.domain.model.metadata.v1.MetadataV1;
import dev.fitko.fitconnect.api.domain.model.metadata.v2.MetadataV2;
import dev.fitko.fitconnect.api.domain.sender.SendableSubmission;
import dev.fitko.fitconnect.api.services.crypto.MessageDigestService;
import dev.fitko.fitconnect.core.crypto.HashService;
import java.util.List;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Utility class for metadata version creation.
 */
public final class MetadataBuilder {

    private static final Logger LOGGER = LoggerFactory.getLogger(MetadataBuilder.class);
    private static final MessageDigestService HASH_SERVICE = new HashService();

    private MetadataBuilder() {
        // Utility class
    }

    /**
     * Creates a complete metadata object for a submission.
     * Determines the appropriate version, creates the metadata instance, and sets all properties.
     *
     * @param submission the submission to create metadata for
     * @param destination the destination to check for supported metadata versions
     * @param uploadedAttachments the list of uploaded attachment payloads
     * @return the complete metadata object ready for submission
     */
    public static Metadata createMetadata(
            SendableSubmission submission, PublicDestination destination, List<AttachmentPayload> uploadedAttachments) {

        final Version version = determineMetadataVersion(submission, destination);
        final Metadata metadata = createMetadataForVersion(version);
        metadata.setSchema(METADATA_SCHEMA_PATH.getSchemaUriForVersion(version).toString());

        final byte[] dataHash = HASH_SERVICE.createHash(submission.getData());
        final String dataHashHexString = HASH_SERVICE.toHexString(dataHash);

        final List<ApiAttachment> apiAttachments = uploadedAttachments.stream()
                .map(AttachmentMapper::toApiAttachment)
                .collect(toList());

        final Data data = createData(dataHashHexString, submission.getSubmissionSchema());

        final var contentStructure = new ContentStructure();
        contentStructure.setData(data);
        contentStructure.setAttachments(apiAttachments);

        metadata.setContentStructure(contentStructure);
        setOptionalProperties(submission, metadata);

        return metadata;
    }

    /**
     * Determines the appropriate metadata version based on destination support.
     * Uses the requested version if supported, otherwise falls back to the latest supported version.
     *
     * @param sendableSubmission the submission with the requested version
     * @param destination the destination to check for supported versions
     * @return the version to use for metadata creation
     * @throws IllegalArgumentException if no valid versions are found in the destination
     */
    public static Version determineMetadataVersion(
            final SendableSubmission sendableSubmission, final PublicDestination destination) {

        final Version requestedVersion = sendableSubmission.getMetadataVersion().getLatestVersion();

        final List<Version> supportedVersions =
                destination.getMetadataVersions().stream().map(Version::new).collect(toList());

        return supportedVersions.stream()
                .filter(version -> version.getMajor() == requestedVersion.getMajor())
                .max(Version::compareTo)
                .orElseGet(useFallBackVersion(supportedVersions));
    }

    private static Supplier<Version> useFallBackVersion(List<Version> supportedVersions) {
        return () -> {
            final Version latestVersion =
                    supportedVersions.stream().max(Version::compareTo).get();
            LOGGER.info(
                    "Requested metadata version is not available on destination, using latest supported version {}",
                    latestVersion.getVersionAsString());
            return latestVersion;
        };
    }

    /**
     * Creates a metadata object for the specified version.
     *
     * @param version the metadata version to create
     * @return the appropriate metadata implementation (MetadataV1 for version 1.x, MetadataV2 for version 2.x)
     * @throws IllegalArgumentException if the version is not supported (only 1.x and 2.x are supported)
     */
    public static Metadata createMetadataForVersion(final Version version) {
        if (version.getMajor() == 1) {
            return new MetadataV1();
        } else if (version.getMajor() == 2) {
            return new MetadataV2();
        }
        throw new IllegalArgumentException(
                "Unsupported metadata version: " + version + ". Supported versions are 1.x and 2.x");
    }

    private static Data createData(final String hashedData, final SubmissionSchema submissionSchema) {
        final var hash = new Hash();
        hash.setContent(hashedData);
        hash.setSignatureType(SignatureType.SHA_512);

        final var data = new Data();
        data.setSubmissionSchema(submissionSchema);
        data.setHash(hash);
        return data;
    }

    private static void setOptionalProperties(final SendableSubmission sendableSubmission, final Metadata metadata) {

        setCommonProperties(sendableSubmission, metadata);

        // Set version-specific properties
        if (metadata instanceof MetadataV1) {
            setMetadataV1Properties(sendableSubmission, (MetadataV1) metadata);
        } else if (metadata instanceof MetadataV2) {
            setMetadataV2Properties(sendableSubmission, (MetadataV2) metadata);
        }
    }

    private static void setCommonProperties(final SendableSubmission sendableSubmission, final Metadata metadata) {
        if (sendableSubmission.getReplyChannel() != null) {
            metadata.setReplyChannel(sendableSubmission.getReplyChannel());
        }

        if (sendableSubmission.getPaymentInformation() != null) {
            metadata.setPaymentInformation(sendableSubmission.getPaymentInformation());
        }

        if (sendableSubmission.getAdditionalReferenceInfo() != null) {
            metadata.setAdditionalReferenceInfo(sendableSubmission.getAdditionalReferenceInfo());
        }
    }

    private static void setMetadataV1Properties(
            final SendableSubmission sendableSubmission, final MetadataV1 metadataV1) {
        if (sendableSubmission.getAuthenticationInformation() != null) {
            metadataV1.setAuthenticationInformation(sendableSubmission.getAuthenticationInformation());
        }
    }

    private static void setMetadataV2Properties(
            final SendableSubmission sendableSubmission, final MetadataV2 metadataV2) {
        if (sendableSubmission.getAuthor() != null) {
            metadataV2.setAuthor(sendableSubmission.getAuthor());
        }

        if (sendableSubmission.getDataSets() != null) {
            metadataV2.setDataSets(sendableSubmission.getDataSets());
        }
    }
}
