package dev.fitko.fitconnect.client.util;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import dev.fitko.fitconnect.api.FitConnectService;
import dev.fitko.fitconnect.api.config.virusscan.VirusScannerMode;
import dev.fitko.fitconnect.api.domain.model.attachment.Attachment;
import dev.fitko.fitconnect.api.domain.model.metadata.Metadata;
import dev.fitko.fitconnect.api.domain.validation.VirusScanResult;
import dev.fitko.fitconnect.api.exceptions.internal.VirusScanException;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Utility class for scanning metadata, data, and attachments for malware using the configured virus
 * scanner.
 */
public class MalwareScanner {

    private static final Logger LOGGER = LoggerFactory.getLogger(MalwareScanner.class);
    private static final ObjectMapper MAPPER = new ObjectMapper();
    private final VirusScannerMode mode;
    private final FitConnectService service;

    /**
     * Creates a new malware scanner with the specified scanning mode and service.
     *
     * @param mode the virus scanning mode to use
     * @param service the FitConnect service for performing virus scans
     */
    public MalwareScanner(VirusScannerMode mode, FitConnectService service) {
        this.mode = mode;
        this.service = service;
    }

    /**
     * Scans metadata for malware by serializing it to bytes and scanning the result.
     *
     * @param metadata the metadata to scan
     * @throws VirusScanException if malware is detected in the metadata or serialization fails
     */
    public VirusScanResult scanMetadata(Metadata metadata) {
        if (mode.equals(VirusScannerMode.NO_OP)) {
            return VirusScanResult.ofClean();
        }
        LOGGER.info("Scanning metadata for malware");
        try {
            final byte[] metadataBytes = MAPPER.writeValueAsBytes(metadata);
            return service.scanBytesForViruses(metadataBytes);
        } catch (JsonProcessingException e) {
            throw new VirusScanException("Failed to serialise metadata object: " + e.getMessage());
        }
    }

    /**
     * Scans raw byte data for malware.
     *
     * @param data the byte array to scan
     * @throws VirusScanException if malware is detected in the data
     */
    public VirusScanResult scanData(byte[] data) {
        if (mode.equals(VirusScannerMode.NO_OP)) {
            return VirusScanResult.ofClean();
        }
        LOGGER.info("Scanning data for malware");
        return service.scanBytesForViruses(data);
    }

    /**
     * Scans a list of attachments for malware.
     *
     * <p>Each attachment is scanned individually, with large attachments scanned from file and small
     * attachments scanned from memory.
     *
     * @param attachments the list of attachments to scan
     * @throws VirusScanException if malware is detected in any attachment
     */
    public VirusScanResult scanAttachments(List<Attachment> attachments) {
        if (mode.equals(VirusScannerMode.NO_OP)) {
            return VirusScanResult.ofClean();
        }
        for (Attachment attachment : attachments) {
            LOGGER.info("Scanning attachment {} for malware", attachment.getAttachmentId());
            final VirusScanResult attachmentScanResult = scanSingleAttachment(attachment);
            if (attachmentScanResult.isInfected()) {
                return attachmentScanResult;
            }
        }
        return VirusScanResult.ofClean();
    }

    /**
     * Scans a single attachment for malware.
     *
     * @param attachment the attachment to scan
     * @return the virus scan result
     */
    private VirusScanResult scanSingleAttachment(Attachment attachment) {
        if (attachment.isLargeAttachment()) {
            return service.scanFileForViruses(attachment.getLargeAttachmentFilePath());
        }
        return service.scanBytesForViruses(attachment.getDataAsBytes());
    }
}
