package dev.fitko.fitconnect.client.util;

import dev.fitko.fitconnect.api.domain.model.attachment.Attachment;
import dev.fitko.fitconnect.api.domain.model.attachment.AttachmentPayload;
import dev.fitko.fitconnect.api.domain.model.attachment.Fragment;
import dev.fitko.fitconnect.api.domain.model.metadata.Hash;
import dev.fitko.fitconnect.api.domain.model.metadata.SignatureType;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.ApiAttachment;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.AttachmentForValidation;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.Purpose;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;

public final class AttachmentMapper {

    private AttachmentMapper() {}

    public static ApiAttachment toApiAttachment(final AttachmentPayload attachmentPayload) {
        final var attachment = new ApiAttachment();
        attachment.setAttachmentId(attachmentPayload.getAttachmentId());
        attachment.setPurpose(attachmentPayload.getPurpose());
        attachment.setFilename(attachmentPayload.getFileName());
        attachment.setMimeType(attachmentPayload.getMimeType());
        attachment.setDescription(attachmentPayload.getDescription());

        if (attachmentPayload.hasFragmentedPayload()) {
            attachment.setFragments(attachmentPayload.getFragmentIds());
        }

        final var hash = new Hash();
        hash.setContent(attachmentPayload.getHashedData());
        hash.setSignatureType(SignatureType.SHA_512);
        attachment.setHash(hash);
        return attachment;
    }

    public static List<Attachment> getReceivedApiAttachments(List<AttachmentForValidation> attachments) {
        return attachments.stream()
                .map(AttachmentMapper::toApiAttachment)
                // omit purpose#data attachments since we only use them to transfer large data that
                // is no actual
                // attachment
                .filter(a -> !a.getPurpose().equals(Purpose.DATA))
                .collect(Collectors.toList());
    }

    public static Attachment toApiAttachment(final AttachmentForValidation attachment) {
        final ApiAttachment metadata = attachment.getAttachmentMetadata();
        if (attachment.hasFragmentedPayload()) {
            return new Attachment(
                    metadata.getAttachmentId(),
                    null,
                    attachment.getDataFile().toPath(),
                    metadata.getMimeType(),
                    metadata.getFilename(),
                    metadata.getDescription(),
                    metadata.getPurpose());
        } else {
            return new Attachment(
                    metadata.getAttachmentId(),
                    attachment.getDecryptedData(),
                    null,
                    metadata.getMimeType(),
                    metadata.getFilename(),
                    metadata.getDescription(),
                    metadata.getPurpose());
        }
    }

    public static Map<UUID, String> mapAttachmentIdsToAuthTags(final List<AttachmentForValidation> attachments) {
        if (attachments == null || attachments.isEmpty()) {
            return Collections.emptyMap();
        }
        final Map<UUID, String> attachmentAuthTags = new LinkedHashMap<>();
        for (final AttachmentForValidation attachment : attachments) {
            if (attachment.hasFragmentedPayload()) {
                attachmentAuthTags.putAll(attachment.getFragments().stream()
                        .collect(Collectors.toMap(Fragment::getFragmentId, Fragment::getAuthTag)));
            } else {
                attachmentAuthTags.put(attachment.getAttachmentId(), attachment.getAuthTag());
            }
        }
        return attachmentAuthTags;
    }

    /**
     * Extract submission data from an attachment with {@link Purpose#DATA}. The data attachment will
     * be removed from the given list.
     *
     * @param attachments to be filtered
     * @return optional byte[] with decrypted submission data
     * @throws IOException if data file could not be read
     */
    public static Optional<byte[]> getSubmissionDataFromAttachments(List<AttachmentForValidation> attachments)
            throws IOException {
        // if there is a Purpose.DATA attachment present, get the submission data from there
        var submissionDataAsAttachment = attachments.stream()
                .filter(AttachmentForValidation::isDataAttachment)
                .findFirst();
        if (submissionDataAsAttachment.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(getDataFromAttachment(submissionDataAsAttachment.get()));
    }

    private static byte[] getDataFromAttachment(AttachmentForValidation attachment) throws IOException {
        if (attachment.hasFragmentedPayload()) {
            final Path path = attachment.getDataFile().toPath();
            final byte[] data = Files.readAllBytes(path);
            // delete data attachment file since we load all data into memory and don't need it
            // anymore
            path.toFile().delete();
            return data;
        }
        return attachment.getDecryptedData();
    }
}
