package dev.fitko.fitconnect.client.subscriber;

import com.nimbusds.jose.jwk.RSAKey;
import dev.fitko.fitconnect.api.FitConnectService;
import dev.fitko.fitconnect.api.config.ApplicationConfig;
import dev.fitko.fitconnect.api.domain.limits.Limit;
import dev.fitko.fitconnect.api.domain.limits.submission.SubmissionLimits;
import dev.fitko.fitconnect.api.domain.model.attachment.Attachment;
import dev.fitko.fitconnect.api.domain.model.attachment.AttachmentPayload;
import dev.fitko.fitconnect.api.domain.model.cases.Case;
import dev.fitko.fitconnect.api.domain.model.destination.PublicDestination;
import dev.fitko.fitconnect.api.domain.model.metadata.ContentStructure;
import dev.fitko.fitconnect.api.domain.model.metadata.Hash;
import dev.fitko.fitconnect.api.domain.model.metadata.Metadata;
import dev.fitko.fitconnect.api.domain.model.metadata.SignatureType;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.Purpose;
import dev.fitko.fitconnect.api.domain.model.metadata.data.Data;
import dev.fitko.fitconnect.api.domain.model.metadata.data.MimeType;
import dev.fitko.fitconnect.api.domain.model.metadata.data.SubmissionSchema;
import dev.fitko.fitconnect.api.domain.model.metadata.v1.MetadataV1;
import dev.fitko.fitconnect.api.domain.model.reply.AnnounceReply;
import dev.fitko.fitconnect.api.domain.model.reply.CreatedReply;
import dev.fitko.fitconnect.api.domain.model.reply.SentReply;
import dev.fitko.fitconnect.api.domain.model.reply.SubmitReply;
import dev.fitko.fitconnect.api.domain.subscriber.SendableReply;
import dev.fitko.fitconnect.api.domain.validation.ValidationResult;
import dev.fitko.fitconnect.api.domain.validation.VirusScanResult;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectReplyException;
import dev.fitko.fitconnect.client.attachments.AttachmentPayloadHandler;
import dev.fitko.fitconnect.client.attachments.upload.AttachmentUploader;
import dev.fitko.fitconnect.client.util.AttachmentMapper;
import dev.fitko.fitconnect.client.util.MalwareScanner;
import dev.fitko.fitconnect.client.util.SubmissionValidator;
import java.util.Collection;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReplySender {

    private static final Logger LOGGER = LoggerFactory.getLogger(ReplySender.class);

    private final ApplicationConfig config;
    private final FitConnectService fitConnectService;
    private final SubmissionValidator submissionValidator;
    private final AttachmentPayloadHandler attachmentPayloadHandler;
    private final AttachmentUploader attachmentUploader;
    private final MalwareScanner malwareScanner;

    public ReplySender(
            final ApplicationConfig config,
            final FitConnectService fitConnectService,
            final SubmissionValidator submissionValidator,
            final AttachmentPayloadHandler attachmentPayloadHandler,
            final AttachmentUploader attachmentUploader) {
        this.config = config;
        this.fitConnectService = fitConnectService;
        this.submissionValidator = submissionValidator;
        this.attachmentPayloadHandler = attachmentPayloadHandler;
        this.attachmentUploader = attachmentUploader;
        this.malwareScanner = new MalwareScanner(config.getVirusScannerMode(), fitConnectService);
    }

    public SentReply sendReply(final SendableReply sendableReply) {

        LOGGER.debug("Scanning data and attachments for malware");
        scanForMalware(sendableReply);

        final Case replyCase = fitConnectService.getCase(sendableReply.getCaseId());
        final PublicDestination destination = fitConnectService.getPublicDestination(replyCase.getDestinationId());
        submissionValidator.ensureValidDataPayload(sendableReply, destination);

        final List<AttachmentPayload> attachmentPayloads = createAttachmentPayloads(sendableReply, destination);

        LOGGER.info("Reply is being created");
        final List<UUID> attachmentIdsToBeAnnounced = getAttachmentIdsToBeAnnounced(attachmentPayloads);
        final AnnounceReply announceReply = new AnnounceReply(sendableReply.getCaseId(), attachmentIdsToBeAnnounced);
        final CreatedReply createdReply = fitConnectService.announceReply(announceReply);

        final RSAKey replyEncryptionKey = sendableReply.getReplyEncryptionKey().toRSAKey();
        LOGGER.info("Uploading {} reply attachment(s)", attachmentPayloads.size());
        List<AttachmentPayload> uploadedAttachments =
                attachmentUploader.uploadAttachments(attachmentPayloads, createdReply.getReplyId(), replyEncryptionKey);

        LOGGER.info("Creating metadata");
        final Metadata metadata = createMetadata(sendableReply, uploadedAttachments);

        final ValidationResult validatedMetadata = fitConnectService.validateMetadataSchema(metadata);
        if (validatedMetadata.hasError()) {
            LOGGER.error("Metadata does not match schema", validatedMetadata.getError());
            throw new FitConnectReplyException(validatedMetadata.getError().getMessage(), validatedMetadata.getError());
        }

        LOGGER.info("Encrypting reply data");
        final String replyDataMimeType = sendableReply.getDataMimeType().value();
        final String encryptedData =
                fitConnectService.encryptBytes(replyEncryptionKey, sendableReply.getData(), replyDataMimeType);

        LOGGER.info("Encrypting reply metadata");
        final String encryptedMetadata =
                fitConnectService.encryptObject(replyEncryptionKey, metadata, MimeType.APPLICATION_JSON.value());

        final SentReply sentReply = fitConnectService.submitReply(
                createdReply.getReplyId(), new SubmitReply(encryptedData, encryptedMetadata));
        LOGGER.info("SUCCESSFULLY SENT REPLY ! \n");
        return sentReply;
    }

    private List<AttachmentPayload> createAttachmentPayloads(
            SendableReply sendableReply, PublicDestination destination) {

        final List<Attachment> attachments = sendableReply.getAttachments();
        if (attachments.stream().anyMatch(a -> a.getPurpose().equals(Purpose.DATA))) {
            LOGGER.info("Reply data exceeds allowed limit and will be sent as attachment");
        }

        if (submissionValidator.destinationSupportsAttachmentChunking(
                destination, attachments, config.getAttachmentChunkingConfig())) {
            final SubmissionLimits caseAttachmentLimits =
                    fitConnectService.getCaseAttachmentLimits(sendableReply.getCaseId());
            final Limit replyAttachmentLimit =
                    caseAttachmentLimits.getAttachments().getReplyLimits().getApplicable();
            return attachmentPayloadHandler.createChunkedPayloads(attachments, replyAttachmentLimit);
        }

        return attachmentPayloadHandler.createPayloadsWithoutChunking(attachments);
    }

    private Metadata createMetadata(final SendableReply sendableReply, final List<AttachmentPayload> attachments) {

        final var hash = new Hash();
        hash.setContent(fitConnectService.createHash(sendableReply.getData()));
        hash.setSignatureType(SignatureType.SHA_512);

        final var submissionSchema =
                new SubmissionSchema(sendableReply.getSchemaUri(), sendableReply.getDataMimeType());

        final var data = new Data();
        data.setSubmissionSchema(submissionSchema);
        data.setHash(hash);

        final var contentStructure = new ContentStructure();
        contentStructure.setAttachments(
                attachments.stream().map(AttachmentMapper::toApiAttachment).collect(Collectors.toList()));
        contentStructure.setData(data);

        final var metadata = new MetadataV1();
        metadata.setSchema(config.getMetadataSchemaWriteVersion().toString());
        metadata.setContentStructure(contentStructure);

        scanForMalware(metadata);

        return metadata;
    }

    private static List<UUID> getAttachmentIdsToBeAnnounced(final List<AttachmentPayload> attachmentPayloads) {
        return attachmentPayloads.stream()
                .map(AttachmentPayload::getAllAttachmentIds)
                .flatMap(Collection::stream)
                .collect(Collectors.toList());
    }

    private void scanForMalware(SendableReply sendableReply) {

        final VirusScanResult dataScanResult = malwareScanner.scanData(sendableReply.getData());
        if (dataScanResult.isInfected()) {
            LOGGER.error("Data is infected with {}", dataScanResult.getSignature());
            throw new FitConnectReplyException("Data is infected with virus: " + dataScanResult.getSignature());
        }

        final VirusScanResult attachmentsScanResult = malwareScanner.scanAttachments(sendableReply.getAttachments());
        if (attachmentsScanResult.isInfected()) {
            LOGGER.error("Attachment is infected with virus {}", attachmentsScanResult.getResult());
            throw new FitConnectReplyException("Attachment "
                    + attachmentsScanResult.getResult()
                    + " is infected with virus: "
                    + attachmentsScanResult.getSignature());
        }
    }

    private void scanForMalware(Metadata metadata) {
        final VirusScanResult dataScanResult = malwareScanner.scanMetadata(metadata);
        if (dataScanResult.isInfected()) {
            LOGGER.error("Metadata is infected with {}", dataScanResult.getSignature());
            throw new FitConnectReplyException("Metadata is infected with virus: " + dataScanResult.getSignature());
        }
    }
}
