package dev.fitko.fitconnect.client.sender;

import static dev.fitko.fitconnect.api.domain.model.event.authtags.AuthenticationTags.getAuthTagFromJWT;
import static dev.fitko.fitconnect.client.util.SubmissionBuilder.*;
import static java.util.stream.Collectors.toMap;

import com.nimbusds.jose.jwk.RSAKey;
import dev.fitko.fitconnect.api.FitConnectService;
import dev.fitko.fitconnect.api.config.ApplicationConfig;
import dev.fitko.fitconnect.api.domain.limits.Limit;
import dev.fitko.fitconnect.api.domain.limits.submission.SubmissionLimits;
import dev.fitko.fitconnect.api.domain.model.attachment.Attachment;
import dev.fitko.fitconnect.api.domain.model.attachment.AttachmentPayload;
import dev.fitko.fitconnect.api.domain.model.attachment.Fragment;
import dev.fitko.fitconnect.api.domain.model.destination.PublicDestination;
import dev.fitko.fitconnect.api.domain.model.event.authtags.AuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.metadata.Metadata;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.Purpose;
import dev.fitko.fitconnect.api.domain.model.metadata.data.MimeType;
import dev.fitko.fitconnect.api.domain.model.submission.*;
import dev.fitko.fitconnect.api.domain.sender.SendableEncryptedSubmission;
import dev.fitko.fitconnect.api.domain.sender.SendableSubmission;
import dev.fitko.fitconnect.api.domain.validation.ValidationResult;
import dev.fitko.fitconnect.api.domain.validation.VirusScanResult;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectSenderException;
import dev.fitko.fitconnect.client.attachments.AttachmentPayloadHandler;
import dev.fitko.fitconnect.client.attachments.upload.AttachmentUploader;
import dev.fitko.fitconnect.client.util.MalwareScanner;
import dev.fitko.fitconnect.client.util.MetadataBuilder;
import dev.fitko.fitconnect.client.util.SubmissionValidator;
import java.util.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SubmissionSender {

    private static final Logger LOGGER = LoggerFactory.getLogger(SubmissionSender.class);
    private final ApplicationConfig config;
    private final FitConnectService fitConnectService;
    private final SubmissionValidator submissionValidator;
    private final AttachmentPayloadHandler attachmentPayloadHandler;
    private final AttachmentUploader attachmentUploader;
    private final MalwareScanner malwareScanner;

    public SubmissionSender(
            final ApplicationConfig config,
            final FitConnectService fitConnectService,
            final SubmissionValidator submissionValidator,
            final AttachmentPayloadHandler attachmentPayloadHandler,
            final AttachmentUploader attachmentUploader) {
        this.config = config;
        this.fitConnectService = fitConnectService;
        this.submissionValidator = submissionValidator;
        this.attachmentPayloadHandler = attachmentPayloadHandler;
        this.attachmentUploader = attachmentUploader;
        this.malwareScanner = new MalwareScanner(config.getVirusScannerMode(), fitConnectService);
    }

    public SentSubmission sendSubmission(final SendableSubmission sendableSubmission) {

        LOGGER.debug("Scanning data and attachments for malware");
        scanForMalware(sendableSubmission);

        final PublicDestination destination =
                fitConnectService.getPublicDestination(sendableSubmission.getDestinationId());
        final RSAKey encryptionKey = fitConnectService.getEncryptionKeyForDestination(destination.getDestinationId());
        submissionValidator.ensureValidDataPayload(sendableSubmission, destination);

        final List<AttachmentPayload> attachmentPayloads = createAttachmentPayloads(sendableSubmission, destination);

        LOGGER.info("Announcing submission");
        final CreatedSubmission announcedSubmission =
                fitConnectService.announceSubmission(buildCreateSubmission(sendableSubmission, attachmentPayloads));

        LOGGER.info("Uploading attachments");
        final List<AttachmentPayload> uploadedAttachments = attachmentUploader.uploadAttachments(
                attachmentPayloads, announcedSubmission.getSubmissionId(), encryptionKey);

        LOGGER.info("Creating metadata with version {}", sendableSubmission.getMetadataVersion());
        final Metadata metadata = MetadataBuilder.createMetadata(sendableSubmission, destination, uploadedAttachments);

        final ValidationResult validatedMetadata = fitConnectService.validateMetadataSchema(metadata);
        if (validatedMetadata.hasError()) {
            LOGGER.error("Metadata does not match schema", validatedMetadata.getError());
            throw new FitConnectSenderException(
                    validatedMetadata.getError().getMessage(), validatedMetadata.getError());
        }

        LOGGER.info("Encrypting submission data ...");
        final String dataMimeType =
                sendableSubmission.getSubmissionSchema().getMimeType().value();
        final String encryptedData =
                fitConnectService.encryptBytes(encryptionKey, sendableSubmission.getData(), dataMimeType);

        LOGGER.info("Encrypting metadata ...");
        final String encryptedMetadata =
                fitConnectService.encryptObject(encryptionKey, metadata, MimeType.APPLICATION_JSON.value());

        final Submission submission = fitConnectService.sendSubmission(
                buildSubmitSubmission(announcedSubmission.getSubmissionId(), encryptedData, encryptedMetadata));
        final AuthenticationTags authenticationTags =
                buildAuthTags(encryptedData, encryptedMetadata, uploadedAttachments);

        LOGGER.info("SUCCESSFULLY HANDED IN SUBMISSION ! \n");
        return buildSentSubmission(submission, authenticationTags);
    }

    public SentSubmission sendEncryptedSubmission(final SendableEncryptedSubmission sendableEncryptedSubmission) {

        final PublicDestination destination =
                fitConnectService.getPublicDestination(sendableEncryptedSubmission.getDestinationId());
        submissionValidator.ensureValidDataPayload(sendableEncryptedSubmission, destination);

        final Map<UUID, String> encryptedAttachments = sendableEncryptedSubmission.getAttachments();
        final String encryptedData = sendableEncryptedSubmission.getData();
        final String encryptedMetadata = sendableEncryptedSubmission.getMetadata();

        final AnnounceSubmission submissionToAnnounce =
                buildCreateSubmission(sendableEncryptedSubmission, encryptedAttachments);
        final UUID submissionId =
                fitConnectService.announceSubmission(submissionToAnnounce).getSubmissionId();
        final SubmitSubmission submitSubmission = buildSubmitSubmission(submissionId, encryptedData, encryptedMetadata);

        uploadEncryptedAttachments(encryptedAttachments, submissionId);

        final Submission submission = fitConnectService.sendSubmission(submitSubmission);

        LOGGER.info("SUCCESSFULLY HANDED IN SUBMISSION ! \n");
        final AuthenticationTags authenticationTags =
                buildAuthTags(encryptedData, encryptedMetadata, encryptedAttachments);
        return buildSentSubmission(submission, authenticationTags);
    }

    private List<AttachmentPayload> createAttachmentPayloads(
            SendableSubmission sendableSubmission, PublicDestination destination) {

        final List<Attachment> attachments = sendableSubmission.getAttachments();
        if (attachments.stream().anyMatch(a -> a.getPurpose().equals(Purpose.DATA))) {
            LOGGER.info("Submission data exceeds allowed limit and will be sent as attachment");
        }

        if (submissionValidator.destinationSupportsAttachmentChunking(
                destination, attachments, config.getAttachmentChunkingConfig())) {
            final SubmissionLimits attachmentLimits =
                    fitConnectService.getDestinationAttachmentLimits(destination.getDestinationId());
            final Limit submissionAttachmentLimit =
                    attachmentLimits.getAttachments().getSubmissionLimits().getApplicable();
            return attachmentPayloadHandler.createChunkedPayloads(attachments, submissionAttachmentLimit);
        }

        return attachmentPayloadHandler.createPayloadsWithoutChunking(attachments);
    }

    public void uploadEncryptedAttachments(final Map<UUID, String> encryptedAttachments, final UUID submissionId) {
        if (encryptedAttachments.isEmpty()) {
            LOGGER.info("No attachments to upload");
        } else {
            LOGGER.info("Uploading {} attachment(s)", encryptedAttachments.size());
            encryptedAttachments.forEach(
                    (key, value) -> fitConnectService.uploadSubmissionAttachment(submissionId, key, value));
        }
    }

    private Map<UUID, String> mapAttachmentIdsToAuthTags(final List<AttachmentPayload> attachmentPayloads) {
        if (attachmentPayloads == null || attachmentPayloads.isEmpty()) {
            return Collections.emptyMap();
        }
        final Map<UUID, String> attachmentAuthTags = new LinkedHashMap<>();
        for (final AttachmentPayload attachment : attachmentPayloads) {
            if (attachment.hasFragmentedPayload()) {
                attachmentAuthTags.putAll(attachment.getFragments().stream()
                        .collect(toMap(Fragment::getFragmentId, Fragment::getAuthTag)));
            } else {
                attachmentAuthTags.put(attachment.getAttachmentId(), attachment.getAuthTag());
            }
        }
        return attachmentAuthTags;
    }

    private Map<UUID, String> mapAttachmentIdsToAuthTags(Map<UUID, String> encryptedAttachments) {
        return encryptedAttachments.entrySet().stream()
                .collect(toMap(Map.Entry::getKey, e -> getAuthTagFromJWT(e.getValue())));
    }

    private AuthenticationTags buildAuthTags(
            final String encryptedData,
            final String encryptedMetadata,
            final List<AttachmentPayload> uploadedAttachments) {
        final String dataAuthTag = getAuthTagFromJWT(encryptedData);
        final String metadataAuthTag = getAuthTagFromJWT(encryptedMetadata);
        final Map<UUID, String> attachmentAuthTags = mapAttachmentIdsToAuthTags(uploadedAttachments);
        return new AuthenticationTags(dataAuthTag, metadataAuthTag, attachmentAuthTags);
    }

    private AuthenticationTags buildAuthTags(
            final String encryptedData, final String encryptedMetadata, final Map<UUID, String> encryptedAttachments) {
        final String dataAuthTag = getAuthTagFromJWT(encryptedData);
        final String metadataAuthTag = getAuthTagFromJWT(encryptedMetadata);
        final Map<UUID, String> attachmentAuthTags = mapAttachmentIdsToAuthTags(encryptedAttachments);
        return new AuthenticationTags(dataAuthTag, metadataAuthTag, attachmentAuthTags);
    }

    private void scanForMalware(SendableSubmission sendableSubmission) {

        final VirusScanResult dataScanResult = malwareScanner.scanData(sendableSubmission.getData());
        if (dataScanResult.isInfected()) {
            LOGGER.error("Data is infected with {}", dataScanResult.getSignature());
            throw new FitConnectSenderException("Data is infected with virus: " + dataScanResult.getSignature());
        }

        final VirusScanResult attachmentsScanResult =
                malwareScanner.scanAttachments(sendableSubmission.getAttachments());
        if (attachmentsScanResult.isInfected()) {
            LOGGER.error("Attachment is infected with virus {}", attachmentsScanResult.getResult());
            throw new FitConnectSenderException("Attachment "
                    + attachmentsScanResult.getResult()
                    + " is infected with virus: "
                    + attachmentsScanResult.getSignature());
        }
    }

    private void scanForMalware(Metadata metadata) {
        final VirusScanResult dataScanResult = malwareScanner.scanMetadata(metadata);
        if (dataScanResult.isInfected()) {
            LOGGER.error("Metadata is infected with {}", dataScanResult.getSignature());
            throw new FitConnectSenderException("Metadata is infected with virus: " + dataScanResult.getSignature());
        }
    }
}
