package dev.fitko.fitconnect.client.sender;

import static com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES;
import static dev.fitko.fitconnect.client.util.AttachmentMapper.getReceivedApiAttachments;
import static dev.fitko.fitconnect.client.util.AttachmentMapper.getSubmissionDataFromAttachments;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.nimbusds.jose.jwk.RSAKey;
import dev.fitko.fitconnect.api.FitConnectService;
import dev.fitko.fitconnect.api.config.ApplicationConfig;
import dev.fitko.fitconnect.api.domain.model.attachment.Attachment;
import dev.fitko.fitconnect.api.domain.model.event.authtags.AuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.event.authtags.ValidatedAuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.event.problems.Problem;
import dev.fitko.fitconnect.api.domain.model.event.problems.data.DataEncryptionIssue;
import dev.fitko.fitconnect.api.domain.model.event.problems.metadata.MetadataEncryptionIssue;
import dev.fitko.fitconnect.api.domain.model.event.problems.metadata.MetadataJsonSyntaxViolation;
import dev.fitko.fitconnect.api.domain.model.metadata.Metadata;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.ApiAttachment;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.AttachmentForValidation;
import dev.fitko.fitconnect.api.domain.model.reply.Reply;
import dev.fitko.fitconnect.api.domain.sender.ReceivedReply;
import dev.fitko.fitconnect.api.domain.sender.ReceivedReplyData;
import dev.fitko.fitconnect.api.domain.validation.ValidationResult;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectReplyException;
import dev.fitko.fitconnect.api.exceptions.internal.DecryptionException;
import dev.fitko.fitconnect.api.exceptions.internal.RestApiException;
import dev.fitko.fitconnect.api.exceptions.internal.SubmissionRequestException;
import dev.fitko.fitconnect.client.attachments.download.AttachmentDownloader;
import dev.fitko.fitconnect.client.util.AttachmentMapper;
import dev.fitko.fitconnect.client.util.MalwareScanner;
import dev.fitko.fitconnect.client.util.MetadataDeserializationHelper;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReplyReceiver {

    private static final Logger LOGGER = LoggerFactory.getLogger(ReplyReceiver.class);

    private static final ObjectMapper MAPPER = new ObjectMapper().configure(FAIL_ON_UNKNOWN_PROPERTIES, false);

    private final ApplicationConfig config;
    private final FitConnectService fitConnectService;
    private final AttachmentDownloader<Reply> attachmentDownloader;
    private final MalwareScanner malwareScanner;

    public ReplyReceiver(
            final ApplicationConfig config,
            final FitConnectService fitConnectService,
            final AttachmentDownloader<Reply> attachmentDownloader) {
        this.config = config;
        this.fitConnectService = fitConnectService;
        this.attachmentDownloader = attachmentDownloader;
        this.malwareScanner = new MalwareScanner(config.getVirusScannerMode(), fitConnectService);
    }

    public ReceivedReply receiveReply(final UUID replyId, final RSAKey replyDecryptionKey) {

        LOGGER.info("Requesting reply");
        final Reply reply = loadReply(replyId);

        LOGGER.info("Loading authentication tags from event log");
        final AuthenticationTags authenticationTagsFromEvent = loadAuthTagsForSubmitEvent(reply);

        LOGGER.info("Decrypting reply metadata");
        final Metadata metadata = decryptMetadata(replyDecryptionKey, reply);
        validateMetadata(reply, authenticationTagsFromEvent, metadata);

        LOGGER.info("Loading reply attachments");
        final List<ApiAttachment> apiAttachments =
                metadata.getContentStructure().getAttachments();
        final List<AttachmentForValidation> attachments =
                attachmentDownloader.loadAttachments(reply, replyDecryptionKey, apiAttachments);
        final AuthenticationTags authenticationTags = buildAuthenticationTags(reply, attachments);
        validateAttachments(reply, attachments, authenticationTags);

        LOGGER.info("Decrypting reply data");
        final byte[] decryptedData = decryptData(replyDecryptionKey, reply, attachments);
        validateData(reply, authenticationTagsFromEvent, metadata, decryptedData);

        return buildReceivedReply(reply, metadata, decryptedData, attachments, authenticationTags);
    }

    private void validateAttachments(
            final Reply reply,
            final List<AttachmentForValidation> attachments,
            final AuthenticationTags authenticationTags) {
        final ValidationResult validatedAttachments =
                fitConnectService.validateAttachments(attachments, authenticationTags);
        if (!validatedAttachments.isValid()) {
            LOGGER.error("Attachment validation failed", validatedAttachments.getError());
            evaluateValidationResult(reply, validatedAttachments);
        }
    }

    private void validateData(
            final Reply reply,
            final AuthenticationTags authenticationTagsFromEvent,
            final Metadata metadata,
            final byte[] decryptedData) {
        final ValidationResult validatedData = fitConnectService.validateData(
                decryptedData, reply.getEncryptedData(), metadata, authenticationTagsFromEvent.getData());
        if (!validatedData.isValid()) {
            LOGGER.error("Data validation failed", validatedData.getError());
            evaluateValidationResult(reply, validatedData);
        }
    }

    private void validateMetadata(
            final Reply reply, final AuthenticationTags authenticationTagsFromEvent, final Metadata metadata) {
        final ValidationResult validatedMetadata =
                fitConnectService.validateReplyMetadata(metadata, reply, authenticationTagsFromEvent);
        if (!validatedMetadata.isValid()) {
            LOGGER.error("Metadata validation failed", validatedMetadata.getError());
            evaluateValidationResult(reply, validatedMetadata);
        }
    }

    private Reply loadReply(final UUID replyId) {
        try {
            return fitConnectService.getReply(replyId);
        } catch (final RestApiException e) {
            throw new FitConnectReplyException(e.getMessage(), e);
        }
    }

    private AuthenticationTags loadAuthTagsForSubmitEvent(final Reply reply) {
        final ValidatedAuthenticationTags validatedAuthenticationTags =
                fitConnectService.getReplyAuthenticationTags(reply);
        if (validatedAuthenticationTags.causesRejection()) {
            rejectReplyWithProblem(reply, validatedAuthenticationTags.getProblem());
            throw new FitConnectReplyException(validatedAuthenticationTags.getErrorMessage());
        }
        return validatedAuthenticationTags.getAuthenticationTags();
    }

    private static AuthenticationTags buildAuthenticationTags(
            final Reply reply, final List<AttachmentForValidation> attachments) {
        final String dataAuthTag = AuthenticationTags.getAuthTagFromJWT(reply.getEncryptedData());
        final String metadataAuthTag = AuthenticationTags.getAuthTagFromJWT(reply.getEncryptedMetadata());
        final Map<UUID, String> attachmentAuthTags = AttachmentMapper.mapAttachmentIdsToAuthTags(attachments);
        return new AuthenticationTags(dataAuthTag, metadataAuthTag, attachmentAuthTags);
    }

    private ReceivedReply buildReceivedReply(
            final Reply reply,
            final Metadata metadata,
            final byte[] decryptedData,
            final List<AttachmentForValidation> attachments,
            final AuthenticationTags authenticationTags) {
        final List<Attachment> receivedAttachments = getReceivedApiAttachments(attachments);
        final ReceivedReplyData receivedData =
                new ReceivedReplyData(decryptedData, metadata, reply, receivedAttachments, authenticationTags);
        return new ReceivedReply(fitConnectService, receivedData);
    }

    private byte[] decryptData(final RSAKey privateKey, final Reply reply, List<AttachmentForValidation> attachments) {
        try {
            return getSubmissionDataFromAttachments(attachments)
                    .orElse(fitConnectService.decryptString(privateKey, reply.getEncryptedData()));
        } catch (final DecryptionException e) {
            // https://docs.fitko.de/fit-connect/docs/receiving/verification/#entschl%C3%BCsselung-1
            reject(reply, List.of(new DataEncryptionIssue()));
            throw new FitConnectReplyException(e.getMessage(), e);
        } catch (IOException e) {
            throw new SubmissionRequestException(e.getMessage(), e);
        }
    }

    private Metadata decryptMetadata(final RSAKey replyDecryptionKey, final Reply reply) {
        try {
            final byte[] metadataBytes =
                    fitConnectService.decryptString(replyDecryptionKey, reply.getEncryptedMetadata());
            return MetadataDeserializationHelper.deserializeMetadata(MAPPER, metadataBytes);
        } catch (final IOException e) {
            reject(reply, List.of(new MetadataJsonSyntaxViolation()));
            throw new FitConnectReplyException(e.getMessage(), e);
        } catch (final DecryptionException e) {
            // https://docs.fitko.de/fit-connect/docs/receiving/verification/#entschl%C3%BCsselung
            reject(reply, List.of(new MetadataEncryptionIssue()));
            throw new FitConnectReplyException(e.getMessage(), e);
        }
    }

    private void evaluateValidationResult(final Reply reply, final ValidationResult validationResult)
            throws SubmissionRequestException {
        if (validationResult.hasProblems()) {
            reject(reply, validationResult.getProblems());
            throw new FitConnectReplyException(
                    validationResult.getProblems().stream()
                            .map(Problem::getDetail)
                            .collect(Collectors.joining()),
                    validationResult.getError());
        }
        LOGGER.error(validationResult.getError().getMessage(), validationResult.getError());
        throw new FitConnectReplyException(validationResult.getError().getMessage(), validationResult.getError());
    }

    private void rejectReplyWithProblem(final Reply reply, final Problem... problem) {
        reject(reply, List.of(problem));
    }

    private void reject(final Reply reply, final List<Problem> problems) {
        if (config.isAutoRejectEnabled()) {
            LOGGER.info(
                    "Auto-rejecting reply due the following problem(s): {}",
                    problems.stream().map(Problem::getDetail).collect(Collectors.joining("\n")));
            fitConnectService.rejectReply(reply.getReplyId(), problems);
        }
    }
}
