package dev.fitko.fitconnect.client.bootstrap;

import static dev.fitko.fitconnect.client.util.ResourceLoadingUtils.loadAllSchemaResources;
import static dev.fitko.fitconnect.client.util.ResourceLoadingUtils.loadYaml;
import static dev.fitko.fitconnect.client.util.ResourceLoadingUtils.readKeyFromPath;
import static java.util.Objects.requireNonNull;

import com.nimbusds.jose.jwk.JWK;
import com.nimbusds.jose.jwk.RSAKey;
import dev.fitko.fitconnect.api.FitConnectService;
import dev.fitko.fitconnect.api.config.ApplicationConfig;
import dev.fitko.fitconnect.api.config.SenderConfig;
import dev.fitko.fitconnect.api.config.SubscriberConfig;
import dev.fitko.fitconnect.api.config.ZBPCertConfig;
import dev.fitko.fitconnect.api.config.chunking.AttachmentChunkingConfig;
import dev.fitko.fitconnect.api.config.defaults.ZBPEnvironment;
import dev.fitko.fitconnect.api.config.http.HttpConfig;
import dev.fitko.fitconnect.api.config.http.ProxyConfig;
import dev.fitko.fitconnect.api.config.http.RetryConfig;
import dev.fitko.fitconnect.api.config.http.Timeouts;
import dev.fitko.fitconnect.api.config.resources.BuildInfo;
import dev.fitko.fitconnect.api.config.resources.CertificateConfig;
import dev.fitko.fitconnect.api.domain.model.reply.Reply;
import dev.fitko.fitconnect.api.domain.model.submission.Submission;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectInitialisationException;
import dev.fitko.fitconnect.api.services.auth.OAuthService;
import dev.fitko.fitconnect.api.services.crypto.CryptoService;
import dev.fitko.fitconnect.api.services.crypto.MessageDigestService;
import dev.fitko.fitconnect.api.services.destination.DestinationService;
import dev.fitko.fitconnect.api.services.events.CaseService;
import dev.fitko.fitconnect.api.services.events.EventLogVerificationService;
import dev.fitko.fitconnect.api.services.events.SecurityEventService;
import dev.fitko.fitconnect.api.services.http.HttpClient;
import dev.fitko.fitconnect.api.services.keys.KeyService;
import dev.fitko.fitconnect.api.services.reply.ReplyService;
import dev.fitko.fitconnect.api.services.routing.RoutingService;
import dev.fitko.fitconnect.api.services.schema.SchemaProvider;
import dev.fitko.fitconnect.api.services.schema.XMLSchemaLoader;
import dev.fitko.fitconnect.api.services.submission.SubmissionService;
import dev.fitko.fitconnect.api.services.validation.ValidationService;
import dev.fitko.fitconnect.api.services.validation.VirusScanService;
import dev.fitko.fitconnect.client.DestinationClient;
import dev.fitko.fitconnect.client.RouterClient;
import dev.fitko.fitconnect.client.SenderClient;
import dev.fitko.fitconnect.client.SubscriberClient;
import dev.fitko.fitconnect.client.ZBPClient;
import dev.fitko.fitconnect.client.attachments.AttachmentPayloadHandler;
import dev.fitko.fitconnect.client.attachments.AttachmentStorageResolver;
import dev.fitko.fitconnect.client.attachments.ConcurrencyLimiter;
import dev.fitko.fitconnect.client.attachments.download.AttachmentDownloader;
import dev.fitko.fitconnect.client.attachments.download.ReplyAttachmentDownloader;
import dev.fitko.fitconnect.client.attachments.download.SubmissionAttachmentDownloader;
import dev.fitko.fitconnect.client.attachments.upload.AttachmentUploader;
import dev.fitko.fitconnect.client.attachments.upload.ReplyAttachmentUploader;
import dev.fitko.fitconnect.client.attachments.upload.SubmissionAttachmentUploader;
import dev.fitko.fitconnect.client.sender.ReplyReceiver;
import dev.fitko.fitconnect.client.sender.SubmissionSender;
import dev.fitko.fitconnect.client.subscriber.ReplySender;
import dev.fitko.fitconnect.client.subscriber.SubmissionReceiver;
import dev.fitko.fitconnect.client.util.ResourceLoadingUtils;
import dev.fitko.fitconnect.client.util.SubmissionValidator;
import dev.fitko.fitconnect.client.zbp.ZBPServiceAdapter;
import dev.fitko.fitconnect.core.FitConnectDefaultService;
import dev.fitko.fitconnect.core.auth.DefaultOAuthApiService;
import dev.fitko.fitconnect.core.cases.CaseApiService;
import dev.fitko.fitconnect.core.cases.EventLogVerifier;
import dev.fitko.fitconnect.core.cases.SecurityEventTokenService;
import dev.fitko.fitconnect.core.crypto.HashService;
import dev.fitko.fitconnect.core.crypto.JWECryptoService;
import dev.fitko.fitconnect.core.destination.DestinationApiService;
import dev.fitko.fitconnect.core.http.DefaultHttpClient;
import dev.fitko.fitconnect.core.http.interceptors.ApiRequestInterceptor;
import dev.fitko.fitconnect.core.http.interceptors.RetryInterceptor;
import dev.fitko.fitconnect.core.http.interceptors.UserAgentInterceptor;
import dev.fitko.fitconnect.core.io.FileChunker;
import dev.fitko.fitconnect.core.keys.PublicKeyApiService;
import dev.fitko.fitconnect.core.reply.ReplyApiService;
import dev.fitko.fitconnect.core.routing.RouteVerifier;
import dev.fitko.fitconnect.core.routing.RoutingApiService;
import dev.fitko.fitconnect.core.schema.SchemaResourceProvider;
import dev.fitko.fitconnect.core.submission.SubmissionApiService;
import dev.fitko.fitconnect.core.validation.DefaultValidationService;
import dev.fitko.fitconnect.core.validation.virusscan.VirusScannerFactory;
import dev.fitko.fitconnect.core.validation.xml.XRepoSchemaLoader;
import dev.fitko.fitconnect.core.validation.xml.XmlSchemaValidator;
import dev.fitko.fitconnect.core.zbp.ZBPApiService;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import okhttp3.Interceptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Factory that constructs clients for multiple aspects of the FIT-Connect ecosystem:
 *
 * <p>
 *
 * <ul>
 *   <li>{@link SenderClient} for handing in submissions
 *   <li>{@link SubscriberClient} for receiving submissions
 *   <li>{@link RouterClient} for finding destinations and areas
 *   <li>{@link DestinationClient} for managing destinations
 *   <li>{@link ZBPClient} for sending messages to BundID mailboxes
 * </ul>
 *
 * @see <a href="https://docs.fitko.de/fit-connect/docs/sdks/java-sdk/overview">SDK-Setup</a>
 * @see <a href="https://docs.fitko.de/fit-connect/docs/sdks/java-sdk/sender">Sending submissions
 *     and receiving replies</a>
 * @see <a href="https://docs.fitko.de/fit-connect/docs/sdks/java-sdk/receiver">Receiving
 *     submissions and sending replies</a>
 * @see <a
 *     href="https://docs.fitko.de/fit-connect/docs/sdks/java-sdk/sender#routing-informationen">Retrieve
 *     routing data</a>
 * @see <a href="https://docs.fitko.de/fit-connect/docs/zbp/">Zentrales Bürgerpostfach (ZBP)</a>
 */
public final class ClientFactory {

    private static final Logger LOGGER = LoggerFactory.getLogger(ClientFactory.class);
    private static final String BUILD_INFO_PATH = "buildinfo.yaml";
    private static final String CERTIFICATE_CONFIG_PATH = "certificate-config.yaml";
    private static final String SCHEMA_CONFIG_PATH = "schema-config.yaml";

    private ClientFactory() {}

    /**
     * Create a new {@link SenderClient} to send submissions and receive replies. This client requires
     * an {@link ApplicationConfig} with a valid {@link SenderConfig}. <br>
     *
     * <pre>{@code
     * final SenderConfig senderConfig = new SenderConfig("CLIENT_ID", "CLIENT_SECRET");
     *
     * final ApplicationConfig config = ApplicationConfig.builder()
     *     .senderConfig(senderConfig)
     *     .activeEnvironment(Environments.TEST.getEnvironmentName())
     *     .build();
     *
     * final SenderClient senderClient = ClientFactory.createSenderClient(config);
     * }</pre>
     *
     * Alternatively the {@link ApplicationConfig} can be loaded from a YAML via {@link
     * ApplicationConfigLoader}
     *
     * @return the {@link SenderClient}
     * @throws FitConnectInitialisationException on errors during the sender client construction
     */
    public static SenderClient createSenderClient(final ApplicationConfig config)
            throws FitConnectInitialisationException {

        requireNonNull(config, "Application config must not be null");

        LOGGER.info("Initializing sender client ...");
        final SenderConfig senderConfig = config.getSenderConfig();
        final FitConnectService fitConnectService =
                createFitConnectService(config, senderConfig.getClientId(), senderConfig.getClientSecret());
        final SubmissionValidator submissionValidator = new SubmissionValidator(fitConnectService);

        final AttachmentChunkingConfig chunkingConfig = config.getAttachmentChunkingConfig();
        final AttachmentStorageResolver attachmentStorageResolver =
                new AttachmentStorageResolver(chunkingConfig.getAttachmentStoragePath());
        final AttachmentPayloadHandler attachmentPayloadHandler =
                createPayloadHandler(chunkingConfig, attachmentStorageResolver);
        final AttachmentUploader attachmentUploader = new SubmissionAttachmentUploader(fitConnectService);
        final AttachmentDownloader<Reply> attachmentDownloader =
                new ReplyAttachmentDownloader(config, fitConnectService, attachmentPayloadHandler);

        final SubmissionSender submissionSender = new SubmissionSender(
                config, fitConnectService, submissionValidator, attachmentPayloadHandler, attachmentUploader);
        final ReplyReceiver replyReceiver = new ReplyReceiver(config, fitConnectService, attachmentDownloader);

        return new SenderClient(fitConnectService, submissionSender, replyReceiver, attachmentStorageResolver);
    }

    /**
     * Create a new {@link SubscriberClient} to receive submissions and send replies. This client
     * requires an {@link ApplicationConfig} with a valid {@link SubscriberConfig}. <br>
     *
     * <pre>{@code
     * final JWK decryptionKey = loadKey("/path/to/decryption_key.json");
     * final JWK signingKey = loadKey("/path/to/signing_key.json");
     *
     * final SubscriberConfig subscriberConfig = SubscriberConfig.builder()
     *     .clientId("CLIENT_ID")
     *     .clientSecret("CLIENT_SECRET")
     *     .privateDecryptionKeys(List.of(decryptionKey))
     *     .privateSigningKeyPath(signingKey)
     *     .build();
     *
     * final ApplicationConfig config = ApplicationConfig.builder()
     *     .subscriberConfig(subscriberConfig)
     *     .activeEnvironment(Environments.TEST.getEnvironmentName())
     *     .build();
     *
     * final SubscriberClient subscriberClient = ClientFactory.createSubscriberClient(config);
     * }</pre>
     *
     * Alternatively the {@link ApplicationConfig} can be loaded from a YAML via {@link
     * ApplicationConfigLoader}
     *
     * @return the {@link SubscriberClient}
     * @throws FitConnectInitialisationException on errors during the subscriber client construction
     */
    public static SubscriberClient createSubscriberClient(final ApplicationConfig config)
            throws FitConnectInitialisationException {

        requireNonNull(config, "Application config must not be null");

        LOGGER.info("Initializing subscriber client ...");
        final SubscriberConfig subscriberConfig = config.getSubscriberConfig();

        final RSAKey rsaDecryptionKey = getDecryptionKeyFromConfig(subscriberConfig);
        final RSAKey rsaSigningKey = getSignatureKeyFromConfig(subscriberConfig);

        final FitConnectService fitConnectService = createFitConnectService(
                config, subscriberConfig.getClientId(), subscriberConfig.getClientSecret(), rsaSigningKey);
        final SubmissionValidator submissionValidator = new SubmissionValidator(fitConnectService);

        final AttachmentChunkingConfig chunkingConfig = config.getAttachmentChunkingConfig();
        final AttachmentStorageResolver attachmentStorageResolver =
                new AttachmentStorageResolver(chunkingConfig.getAttachmentStoragePath());
        final AttachmentPayloadHandler attachmentPayloadHandler =
                createPayloadHandler(chunkingConfig, attachmentStorageResolver);
        final AttachmentUploader attachmentUploader = new ReplyAttachmentUploader(fitConnectService);
        final AttachmentDownloader<Submission> attachmentDownloader =
                new SubmissionAttachmentDownloader(config, fitConnectService, attachmentPayloadHandler);

        final SubmissionReceiver submissionReceiver =
                new SubmissionReceiver(fitConnectService, rsaDecryptionKey, config, attachmentDownloader);
        final ReplySender replySender = new ReplySender(
                config, fitConnectService, submissionValidator, attachmentPayloadHandler, attachmentUploader);

        return new SubscriberClient(fitConnectService, submissionReceiver, replySender, attachmentStorageResolver);
    }

    /**
     * Create a new {@link RouterClient} to find destinations and services. This client requires an
     * {@link ApplicationConfig} without credentials for sender-config or subscriber-config, since
     * routing doesn't need authentication. <br>
     *
     * <pre>{@code
     * final ApplicationConfig config = ApplicationConfig.builder()
     *     .activeEnvironment(Environments.TEST.getEnvironmentName())
     *     .build();
     *
     * final RouterClient routerClient = ClientFactory.createRouterClient(config);
     * }</pre>
     *
     * Alternatively the {@link ApplicationConfig} can be loaded from a YAML via {@link
     * ApplicationConfigLoader}
     *
     * @return the {@link RouterClient}
     * @throws FitConnectInitialisationException on errors during the routing client construction
     */
    public static RouterClient createRouterClient(final ApplicationConfig config)
            throws FitConnectInitialisationException {

        requireNonNull(config, "Application config must not be null");

        LOGGER.info("Initializing router client ...");
        final HttpClient httpClient = createHttpClient(config);
        final SchemaProvider schemaProvider = createSchemaProvider(httpClient, config);
        final ValidationService validator =
                createValidationService(config, schemaProvider, new HashService(), httpClient);

        final KeyService keyService = new PublicKeyApiService(config, httpClient, validator);
        final RouteVerifier routeVerifier = new RouteVerifier(keyService, validator);
        final RoutingService routingService = new RoutingApiService(httpClient, config.getRoutingBaseUrl());

        return new RouterClient(routingService, routeVerifier);
    }

    /**
     * Create a new {@link DestinationClient} to create and manage destinations. This client requires
     * an {@link ApplicationConfig} with a valid {@link SubscriberConfig}. <br>
     *
     * <pre>{@code
     * final JWK decryptionKey = loadKey("/path/to/decryption_key.json");
     * final JWK signingKey = loadKey("/path/to/signing_key.json");
     *
     * final SubscriberConfig subscriberConfig = SubscriberConfig.builder()
     *     .clientId("CLIENT_ID")
     *     .clientSecret("CLIENT_SECRET")
     *     .privateDecryptionKeys(List.of(decryptionKey))
     *     .privateSigningKeyPath(signingKey)
     *     .build();
     *
     * final ApplicationConfig config = ApplicationConfig.builder()
     *     .subscriberConfig(subscriberConfig)
     *     .activeEnvironment(Environments.TEST.getEnvironmentName())
     *     .build();
     *
     * final DestinationClient destinationClient = ClientFactory.createDestinationClient(config);
     * }</pre>
     *
     * Alternatively the {@link ApplicationConfig} can be loaded from a YAML via {@link
     * ApplicationConfigLoader}
     *
     * @return the {@link DestinationClient}
     * @throws FitConnectInitialisationException on errors during the destination client construction
     */
    public static DestinationClient createDestinationClient(ApplicationConfig config)
            throws FitConnectInitialisationException {

        requireNonNull(config, "Application config must not be null");
        requireNonNull(config.getSubscriberConfig(), "DestinationClient requires a subscriber config, but was null");
        LOGGER.info("Initializing destination client ...");

        final SubscriberConfig subscriberConfig = config.getSubscriberConfig();
        final String clientId = subscriberConfig.getClientId();
        final String clientSecret = subscriberConfig.getClientSecret();

        final var httpClient = createHttpClient(config);
        final var authService = new DefaultOAuthApiService(httpClient, clientId, clientSecret, config.getAuthBaseUrl());
        final var destinationApiService =
                new DestinationApiService(authService, httpClient, config.getDestinationBaseUrl());
        return new DestinationClient(destinationApiService);
    }

    /**
     * Create a new client to send messages and status updates to mailboxes of the ZBP (Zentrales
     * Bürgerpostfach). <br>
     *
     * <pre>{@code
     * final var privateKeyPath = Path.of("/path/to/clientPrivateKey.key");
     * final var clientCertPath = Path.of("/path/to/clientCertificate.crt");
     *
     * final ZBPCertConfig certConfig = new ZBPCertConfig(privateKeyPath, clientCertPath)
     * final ZBPClient zbpClient = ClientFactory.createZBPClient(certConfig, ZBPEnvironment.INT_INTERNET);
     * }</pre>
     *
     * @param certConfig the certificate configuration holding the private signing key and the public
     *     key
     * @param environment the environment requests are sent to
     * @return the {@link ZBPClient}
     * @throws FitConnectInitialisationException on errors during the zbp client construction
     */
    public static ZBPClient createZBPClient(ZBPCertConfig certConfig, ZBPEnvironment environment) {
        return createZBPClient(certConfig, environment, HttpConfig.builder().build());
    }

    /**
     * Create a new client with an additional {@link HttpConfig}, to send messages and status updates
     * to mailboxes of the ZBP (Zentrales Bürgerpostfach). <br>
     *
     * <pre>{@code
     * final var privateKeyPath = Path.of("/path/to/clientPrivateKey.key");
     * final var clientCertPath = Path.of("/path/to/clientCertificate.crt");
     *
     * final RetryConfig retryConfig = RetryConfig.builder()
     *     .initialDelayInMs(100)
     *     .maxRetryCount(3)
     *     .build();
     *
     * final HttpConfig httpConfig = HttpConfig.builder()
     *     .retryConfig(retryConfig)
     *     .build();
     *
     * final ZBPCertConfig certConfig = new ZBPCertConfig(privateKeyPath, clientCertPath)
     * final ZBPClient zbpClient = ClientFactory.createZBPClient(certConfig, ZBPEnvironment.INT_INTERNET, httpConfig);
     * }</pre>
     *
     * @param certConfig the certificate configuration holding the private signing key and the public
     *     key
     * @param environment the environment requests are sent to
     * @param httpConfig HTTP-client configuration to pass custom {@link RetryConfig}, {@link
     *     Timeouts} or {@link ProxyConfig} settings
     * @return the {@link ZBPClient}
     * @throws FitConnectInitialisationException on errors during the zbp client construction
     */
    public static ZBPClient createZBPClient(
            ZBPCertConfig certConfig, ZBPEnvironment environment, HttpConfig httpConfig) {
        LOGGER.info("Initializing ZBP client ...");
        final ApplicationConfig config =
                ApplicationConfig.builder().httpConfig(httpConfig).build();
        final HttpClient httpClient = createHttpClient(config, certConfig, environment);
        final ZBPApiService apiService = new ZBPApiService(httpClient, environment, certConfig);
        return new ZBPClient(new ZBPServiceAdapter(apiService, certConfig));
    }

    private static FitConnectService createFitConnectService(
            final ApplicationConfig config, final String clientId, final String clientSecret) {
        return createFitConnectService(config, clientId, clientSecret, null);
    }

    private static FitConnectService createFitConnectService(
            final ApplicationConfig config, final String clientId, final String clientSecret, final RSAKey signingKey) {
        final HttpClient httpClient = createHttpClient(config);
        final SchemaProvider schemaProvider = createSchemaProvider(httpClient, config);
        final MessageDigestService messageDigestService = new HashService();

        final CryptoService cryptoService = new JWECryptoService(messageDigestService);
        final ValidationService validator =
                createValidationService(config, schemaProvider, messageDigestService, httpClient);

        final OAuthService authService =
                new DefaultOAuthApiService(httpClient, clientId, clientSecret, config.getAuthBaseUrl());

        final SubmissionService submissionService =
                new SubmissionApiService(authService, httpClient, config.getSubmissionBaseUrl());
        final DestinationService destinationService =
                new DestinationApiService(authService, httpClient, config.getDestinationBaseUrl());

        final KeyService keyService = new PublicKeyApiService(config, httpClient, authService, validator);
        final EventLogVerificationService eventLogVerifier = new EventLogVerifier(keyService, validator);
        final CaseService caseService =
                new CaseApiService(authService, httpClient, eventLogVerifier, config.getSubmissionBaseUrl());
        final ReplyService replyService = new ReplyApiService(authService, httpClient, config.getSubmissionBaseUrl());

        final SecurityEventService setService = new SecurityEventTokenService(config, validator, signingKey);
        final ConcurrencyLimiter concurrencyLimiter = new ConcurrencyLimiter(config.getConcurrentAttachmentStreams());

        final VirusScanService virusScanner =
                VirusScannerFactory.createScanner(config.getVirusScannerMode(), config.getVirusScannerConfig());

        return FitConnectDefaultService.builder()
                .submissionService(submissionService)
                .destinationService(destinationService)
                .caseService(caseService)
                .replyService(replyService)
                .cryptoService(cryptoService)
                .validationService(validator)
                .keyService(keyService)
                .securityEventService(setService)
                .concurrencyLimiter(concurrencyLimiter)
                .virusScanService(virusScanner)
                .build();
    }

    private static ValidationService createValidationService(
            final ApplicationConfig config,
            final SchemaProvider schemaProvider,
            final MessageDigestService messageDigestService,
            HttpClient httpClient) {
        final List<String> trustedRootCertificates =
                loadYaml(CERTIFICATE_CONFIG_PATH, CertificateConfig.class).getTrustedRootCertificates().stream()
                        .map(ResourceLoadingUtils::readResourceToString)
                        .collect(Collectors.toList());
        LOGGER.info("Initialised trusted root certificates");
        final XmlSchemaValidator xmlSchemaValidator = createXmlSchemaValidator(httpClient);
        return new DefaultValidationService(
                config, messageDigestService, schemaProvider, xmlSchemaValidator, trustedRootCertificates);
    }

    private static XmlSchemaValidator createXmlSchemaValidator(HttpClient httpClient) {
        final XMLSchemaLoader schemaLoader = new XRepoSchemaLoader(httpClient);
        return new XmlSchemaValidator(schemaLoader);
    }

    private static HttpClient createHttpClient(final ApplicationConfig config) {
        return createHttpClient(config, null, null);
    }

    private static HttpClient createHttpClient(
            final ApplicationConfig config, ZBPCertConfig zbpCertConfig, ZBPEnvironment zbpEnvironment) {
        final BuildInfo buildInfo = loadYaml(BUILD_INFO_PATH, BuildInfo.class);
        final HttpConfig httpConfig = config.getHttpConfig();
        final List<Interceptor> interceptors = getInterceptors(buildInfo, httpConfig);
        return new DefaultHttpClient(httpConfig, interceptors, zbpCertConfig, zbpEnvironment);
    }

    private static List<Interceptor> getInterceptors(BuildInfo buildInfo, HttpConfig httpConfig) {
        final List<Interceptor> interceptors = new ArrayList<>();
        interceptors.add(new ApiRequestInterceptor());
        interceptors.add(new UserAgentInterceptor(buildInfo));
        final RetryConfig retryConfig = httpConfig.getRetryConfig();
        if (retryConfig.isAllowRetries()) {
            LOGGER.info("Using {} retries for failed HTTP-calls", retryConfig.getMaxRetryCount());
            interceptors.add(new RetryInterceptor(retryConfig));
        }
        return interceptors;
    }

    private static SchemaProvider createSchemaProvider(final HttpClient httpClient, final ApplicationConfig config) {
        final var schemaResources = loadAllSchemaResources(SCHEMA_CONFIG_PATH, config.getSubmissionDataSchemas());
        return new SchemaResourceProvider(httpClient, schemaResources);
    }

    private static AttachmentPayloadHandler createPayloadHandler(
            final AttachmentChunkingConfig config, AttachmentStorageResolver attachmentStorageResolver) {
        final FileChunker fileChunker = new FileChunker();
        final MessageDigestService messageDigestService = new HashService();
        return new AttachmentPayloadHandler(config, fileChunker, attachmentStorageResolver, messageDigestService);
    }

    private static RSAKey parseRSAKeyFromString(final String key) {
        try {
            return RSAKey.parse(JWK.parse(key).toJSONObject());
        } catch (final ParseException | NullPointerException e) {
            LOGGER.error("Could not parse key, please check environment (config.yml)");
            throw new FitConnectInitialisationException("Key could not be parsed", e);
        }
    }

    private static String getPrivateDecryptionKeyPathFromSubscriber(final SubscriberConfig subscriberConfig) {
        if (subscriberConfig.getPrivateDecryptionKeyPaths().size() != 1) {
            throw new FitConnectInitialisationException(
                    "Currently only one configured private key per subscriber is allowed !");
        }
        return subscriberConfig.getPrivateDecryptionKeyPaths().get(0);
    }

    private static RSAKey getSignatureKeyFromConfig(SubscriberConfig config) {
        LOGGER.info("Initialising private signature key");
        if (config.getSubscriberKeys() != null) {
            return config.getSubscriberKeys().getPrivateSigningKey().toRSAKey();
        }
        return parseRSAKeyFromString(readKeyFromPath(config.getPrivateSigningKeyPath()));
    }

    private static RSAKey getDecryptionKeyFromConfig(SubscriberConfig config) {
        LOGGER.info("Initialising private decryption key");
        if (config.getSubscriberKeys() != null) {
            return config.getSubscriberKeys().getPrivateDecryptionKeys().get(0).toRSAKey();
        }
        final String decryptionKeyPath = getPrivateDecryptionKeyPathFromSubscriber(config);
        final String decryptionKey = readKeyFromPath(decryptionKeyPath);
        return parseRSAKeyFromString(decryptionKey);
    }
}
