package dev.fitko.fitconnect.client.attachments.upload;

import static dev.fitko.fitconnect.api.domain.model.event.authtags.AuthenticationTags.getAuthTagFromJWT;

import com.nimbusds.jose.JWEObject;
import com.nimbusds.jose.jwk.RSAKey;
import dev.fitko.fitconnect.api.FitConnectService;
import dev.fitko.fitconnect.api.domain.model.attachment.AttachmentPayload;
import dev.fitko.fitconnect.api.domain.model.attachment.Fragment;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AttachmentUploader {

    private static final Logger LOGGER = LoggerFactory.getLogger(AttachmentUploader.class);

    protected final FitConnectService fitConnectService;

    protected AttachmentUploader(final FitConnectService fitConnectService) {
        this.fitConnectService = fitConnectService;
    }

    /**
     * Upload attachment payload for a given transmission (submission or reply).
     *
     * @param attachmentPayloads list of attachment payloads to be uploaded
     * @param transmissionId unique identifier of the transmission (submission | reply)
     * @param encryptionKey the public encryption key the payload is encrypted with
     * @return list of uploaded payloads with data that is encrypted, hashed and has generated
     *     auth-tags
     */
    public List<AttachmentPayload> uploadAttachments(
            final List<AttachmentPayload> attachmentPayloads, final UUID transmissionId, final RSAKey encryptionKey) {
        if (attachmentPayloads.isEmpty()) {
            LOGGER.info("No attachments to upload");
            return attachmentPayloads;
        }
        LOGGER.info("Uploading {} attachment(s)", attachmentPayloads.size());
        final List<AttachmentPayload> uploadedAttachments = new ArrayList<>();
        attachmentPayloads.forEach(attachment -> {
            if (attachment.hasFragmentedPayload()) {
                uploadedAttachments.add(uploadFragments(transmissionId, encryptionKey, attachment));
            } else {
                uploadedAttachments.add(uploadSingleAttachment(transmissionId, encryptionKey, attachment));
            }
        });
        return uploadedAttachments;
    }

    private AttachmentPayload uploadSingleAttachment(
            final UUID transmissionId, final RSAKey encryptionKey, final AttachmentPayload attachment) {
        final String encryptedData =
                fitConnectService.encryptBytes(encryptionKey, attachment.getData(), attachment.getMimeType());
        final String hashedData = fitConnectService.createHash(attachment.getData());
        upload(transmissionId, attachment.getAttachmentId(), encryptedData);
        return attachment.withHashedData(hashedData).withAuthTag(getAuthTagFromJWT(encryptedData));
    }

    private AttachmentPayload uploadFragments(UUID transmissionId, RSAKey encryptionKey, AttachmentPayload attachment) {
        LOGGER.info(
                "Uploading {} fragments for attachment {}",
                attachment.getFragments().size(),
                attachment.getAttachmentId());
        for (Fragment fragment : attachment.getFragments()) {
            uploadSingleFragment(transmissionId, encryptionKey, attachment.getMimeType(), fragment);
        }
        attachment.getFragmentsBaseFolder().ifPresent(File::delete);
        return attachment;
    }

    private void uploadSingleFragment(
            final UUID transmissionId, final RSAKey encryptionKey, final String mimeType, final Fragment fragment) {
        try (InputStream is = new FileInputStream(fragment.getFile())) {
            final JWEObject encryptedData = fitConnectService.encryptInputStream(encryptionKey, is, mimeType);
            upload(transmissionId, fragment.getFragmentId(), encryptedData.serialize());
            fragment.setAuthTag(encryptedData.getAuthTag().toString());
        } catch (final IOException e) {
            throw new RuntimeException(e);
        } finally {
            fragment.getFile().delete();
        }
    }

    protected abstract void upload(UUID transmissionId, UUID attachmentId, String encryptedData);
}
