package dev.fitko.fitconnect.client.attachments.download;

import com.nimbusds.jose.jwk.RSAKey;
import dev.fitko.fitconnect.api.FitConnectService;
import dev.fitko.fitconnect.api.domain.model.attachment.Fragment;
import dev.fitko.fitconnect.api.domain.model.event.authtags.AuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.event.problems.Problem;
import dev.fitko.fitconnect.api.domain.model.event.problems.attachment.AttachmentEncryptionIssue;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.ApiAttachment;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.AttachmentForValidation;
import dev.fitko.fitconnect.api.exceptions.internal.DecryptionException;
import dev.fitko.fitconnect.api.exceptions.internal.SubmissionRequestException;
import dev.fitko.fitconnect.client.attachments.AttachmentPayloadHandler;
import dev.fitko.fitconnect.core.utils.StopWatch;
import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.IntStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AttachmentDownloader<T> {

    protected static final Logger LOGGER = LoggerFactory.getLogger(AttachmentDownloader.class);

    protected final FitConnectService fitConnectService;
    private final AttachmentPayloadHandler attachmentPayloadHandler;

    protected AttachmentDownloader(
            final FitConnectService fitConnectService, final AttachmentPayloadHandler attachmentPayloadHandler) {
        this.fitConnectService = fitConnectService;
        this.attachmentPayloadHandler = attachmentPayloadHandler;
    }

    /**
     * Download attachments for a given transmission (submission or reply).
     *
     * @param transmission unique identifier of the transmission (submission | reply)
     * @param privateDecryptionKey private decryption key the payload is decrypted with
     * @param apiAttachments list of api-attachment metadata
     * @return list of decrypted attachments ready for validation
     */
    public List<AttachmentForValidation> loadAttachments(
            final T transmission, final RSAKey privateDecryptionKey, final List<ApiAttachment> apiAttachments) {
        if (apiAttachments == null || apiAttachments.isEmpty()) {
            LOGGER.info("Submission contains no attachments");
            return Collections.emptyList();
        }
        final List<AttachmentForValidation> receivedAttachments = new ArrayList<>();
        for (final ApiAttachment apiAttachment : apiAttachments) {
            if (apiAttachment.hasFragments()) {
                receivedAttachments.add(loadFragmentedAttachment(transmission, privateDecryptionKey, apiAttachment));
            } else {
                receivedAttachments.add(
                        loadAttachmentWithoutFragments(transmission, privateDecryptionKey, apiAttachment));
            }
        }
        return receivedAttachments;
    }

    private AttachmentForValidation loadFragmentedAttachment(
            final T transmission, final RSAKey privateDecryptionKey, final ApiAttachment apiAttachment) {
        final List<Fragment> fragments = downloadFragments(transmission, privateDecryptionKey, apiAttachment);
        final Optional<File> mergedFile = attachmentPayloadHandler.mergeChunks(apiAttachment);
        return AttachmentForValidation.forFragmentedAttachment(apiAttachment, fragments, mergedFile.orElse(null));
    }

    private AttachmentForValidation loadAttachmentWithoutFragments(
            final T transmission, final RSAKey privateDecryptionKey, final ApiAttachment apiAttachment) {
        final UUID attachmentId = apiAttachment.getAttachmentId();
        final String encryptedAttachment = downloadAttachment(transmission, attachmentId);
        final byte[] decryptedAttachment =
                decryptAttachment(privateDecryptionKey, attachmentId, encryptedAttachment, transmission);
        final String authTag = AuthenticationTags.getAuthTagFromJWT(encryptedAttachment);
        return AttachmentForValidation.forAttachmentWithoutFragments(apiAttachment, decryptedAttachment, authTag);
    }

    private List<Fragment> downloadFragments(
            final T transmission, final RSAKey privateDecryptionKey, final ApiAttachment apiAttachment) {
        final List<Fragment> downloadedFragments = new ArrayList<>();
        final List<UUID> fragments = apiAttachment.getFragments();
        IntStream.range(0, fragments.size()).forEachOrdered(index -> {
            final UUID fragmentId = fragments.get(index);
            final UUID attachmentId = apiAttachment.getAttachmentId();
            downloadedFragments.add(
                    downloadFragment(transmission, privateDecryptionKey, attachmentId, fragmentId, index));
        });
        return downloadedFragments;
    }

    private Fragment downloadFragment(
            final T transmission,
            final RSAKey privateDecryptionKey,
            final UUID parentAttachmentId,
            final UUID fragmentId,
            final int fragmentIndex) {
        final String encryptedAttachment = downloadAttachment(transmission, fragmentId);
        final byte[] decryptedAttachment =
                decryptAttachment(privateDecryptionKey, fragmentId, encryptedAttachment, transmission);
        final String authTag = AuthenticationTags.getAuthTagFromJWT(encryptedAttachment);
        final File file = attachmentPayloadHandler.writeReceivedFragmentToFileSystem(
                parentAttachmentId, fragmentIndex, decryptedAttachment);
        return new Fragment(fragmentId, authTag, file);
    }

    private byte[] decryptAttachment(
            final RSAKey privateDecryptionKey,
            final UUID attachmentId,
            final String encryptedAttachment,
            final T transmission) {
        try {
            final var startDecryption = StopWatch.start();
            final byte[] decryptedAttachment =
                    fitConnectService.decryptString(privateDecryptionKey, encryptedAttachment);
            LOGGER.trace("Decrypting attachment {} took {}", attachmentId, StopWatch.stop(startDecryption));
            return decryptedAttachment;
        } catch (final DecryptionException e) {
            // https://docs.fitko.de/fit-connect/docs/receiving/verification/#entschl%C3%BCsselung-2
            reject(transmission, new AttachmentEncryptionIssue(attachmentId));
            throw new SubmissionRequestException(e.getMessage(), e);
        }
    }

    protected abstract String downloadAttachment(T transmission, UUID id);

    protected abstract void reject(T transmission, Problem problem);
}
