package dev.fitko.fitconnect.client.attachments;

import static dev.fitko.fitconnect.core.utils.Preconditions.checkArgumentAndThrow;

import java.util.concurrent.Semaphore;

/**
 * An instance of this class can be used to allow only N threads to execute a specific code section
 * concurrently.
 */
public class ConcurrencyLimiter {

    private final Semaphore semaphore;

    /**
     * @param concurrencyLimit the number of threads allowed to work at the same time
     */
    public ConcurrencyLimiter(int concurrencyLimit) {
        checkArgumentAndThrow(concurrencyLimit <= 0, "concurrencyLimit must be > 0");
        this.semaphore = new Semaphore(concurrencyLimit, true);
    }

    /**
     * Blocks until the calling thread is allowed to enter the concurrency-limited section. The caller
     * MUST call close() on the returned object once when the work is done, for example using try-with
     * syntax.
     *
     * @return auto-closable object
     */
    public AcquiredSemaphore acquire() {
        try {
            semaphore.acquire();
            return new AcquiredSemaphoreImpl();
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    public interface AcquiredSemaphore extends AutoCloseable {
        void close() throws RuntimeException;
    }

    private class AcquiredSemaphoreImpl implements AcquiredSemaphore {

        @Override
        public void close() {
            semaphore.release();
        }
    }
}
