package dev.fitko.fitconnect.client.attachments;

import static dev.fitko.fitconnect.api.config.chunking.AttachmentChunkingConfig.DEFAULT_ATTACHMENT_FOLDER_NAME;
import static java.util.Comparator.reverseOrder;

import dev.fitko.fitconnect.api.config.chunking.AttachmentChunkingConfig;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectAttachmentException;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Predicate;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AttachmentStorageResolver {

    public static final String TMP_DIR = System.getProperty("java.io.tmpdir");
    private static final Logger LOGGER = LoggerFactory.getLogger(AttachmentStorageResolver.class);
    private static final Predicate<Path> IS_INCOMING_PATH =
            path -> path.toString().contains(Direction.INCOMING.getName());

    private final Path basePath;

    /**
     * Constructs a base path from the given path. If null, a temp-dir will be created.
     *
     * @param attachmentStoragePath path that is set as storage base path
     */
    public AttachmentStorageResolver(final Path attachmentStoragePath) {
        this.basePath = Optional.ofNullable(attachmentStoragePath).orElse(createTempDir());
        LOGGER.info("Configured {} as attachment storage path", basePath);
    }

    /**
     * Constructs a base path in the default temporary-file directory.
     *
     * <p>{@see AttachmentChunkingConfig#DEFAULT_TEMP_ATTACHMENT_FOLDER_NAME}
     */
    public AttachmentStorageResolver() {
        this(null);
    }

    /**
     * Resolve incoming attachment folder for a given attachmentId within the base path.
     *
     * @param attachmentId unique identifier of the attachment a folder within the base path is
     *     created
     * @return path of attachment folder
     * @throws IOException if an error occurred during the path creation
     * @see AttachmentChunkingConfig#getAttachmentStoragePath()
     */
    public Path resolveIncomingAttachmentFolder(final UUID attachmentId) throws IOException {
        return resolveFolder(Direction.INCOMING, attachmentId);
    }

    /**
     * Resolve outgoing attachment folder for a given attachmentId within the base path.
     *
     * @param attachmentId unique identifier of the attachment a folder within the base path is
     *     created
     * @return path of attachment folder
     * @throws IOException if an error occurred during the path creation
     * @see AttachmentChunkingConfig#getAttachmentStoragePath()
     */
    public Path resolveOutgoingAttachmentFolder(final UUID attachmentId) throws IOException {
        return resolveFolder(Direction.OUTGOING, attachmentId);
    }

    /**
     * Removes all incoming files and folders within the base path.
     *
     * @throws FitConnectAttachmentException if the folder cannot be accessed or a file cannot be
     *     deleted
     */
    public void clearBasePathContents() {
        LOGGER.info("Clearing attachment storage path {}", basePath);
        try (Stream<Path> fileTree = Files.walk(basePath).sorted(reverseOrder())) {
            fileTree.filter(IS_INCOMING_PATH).forEach(AttachmentStorageResolver::deletePath);
        } catch (IOException | RuntimeException e) {
            throw new FitConnectAttachmentException(e.getMessage(), e);
        }
    }

    private Path resolveFolder(Direction direction, final UUID id) throws IOException {
        return Files.createDirectories(this.basePath.resolve(direction.name + id));
    }

    private Path createTempDir() {
        try {
            return Files.createDirectories(Path.of(TMP_DIR).resolve(DEFAULT_ATTACHMENT_FOLDER_NAME));
        } catch (final IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static void deletePath(Path path) {
        try {
            LOGGER.debug("Deleting {} {}", path.toFile().isDirectory() ? "directory" : "file", path);
            Files.delete(path);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public enum Direction {
        INCOMING("INCOMING_"),
        OUTGOING("OUTGOING_");
        private final String name;

        Direction(final String name) {
            this.name = name;
        }

        public String getName() {
            return name;
        }
    }
}
