package dev.fitko.fitconnect.client.attachments;

import static dev.fitko.fitconnect.api.config.chunking.AttachmentChunkingConfig.TEMP_BUFFERED_FILE_PREFIX;
import static dev.fitko.fitconnect.core.utils.Formatter.toHumanReadableSizePrefix;
import static dev.fitko.fitconnect.core.utils.Formatter.toMegaBytes;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Stream.concat;

import dev.fitko.fitconnect.api.config.chunking.AttachmentChunkingConfig;
import dev.fitko.fitconnect.api.config.chunking.ChunkSize;
import dev.fitko.fitconnect.api.domain.limits.Limit;
import dev.fitko.fitconnect.api.domain.model.attachment.Attachment;
import dev.fitko.fitconnect.api.domain.model.attachment.AttachmentPayload;
import dev.fitko.fitconnect.api.domain.model.attachment.Fragment;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.ApiAttachment;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.Purpose;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectAttachmentException;
import dev.fitko.fitconnect.api.services.crypto.MessageDigestService;
import dev.fitko.fitconnect.core.io.FileChunker;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Function;
import java.util.function.Predicate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** Creates fragmented and un-fragmented attachment payloads. */
public class AttachmentPayloadHandler {

    private static final Logger LOGGER = LoggerFactory.getLogger(AttachmentPayloadHandler.class);
    public static final double ENCRYPTION_OVERHEAD_BUFFER_FACTOR = 0.6;

    private ChunkSize chunkSize;
    private final AttachmentChunkingConfig config;
    private final FileChunker chunker;
    private final MessageDigestService hashService;
    private final AttachmentStorageResolver attachmentStorageResolver;

    public AttachmentPayloadHandler(
            final AttachmentChunkingConfig config,
            final FileChunker chunker,
            final AttachmentStorageResolver attachmentStorageResolver,
            final MessageDigestService hashService) {
        this.config = config;
        this.chunker = chunker;
        this.hashService = hashService;
        this.chunkSize = config.getChunkSizeObject();
        this.attachmentStorageResolver = attachmentStorageResolver;
    }

    /**
     * Transforms a list of attachments into payloads of multiple chunks. If the configured chunk-size
     * is null or greater than the allowed limit, the limit will be used as the default chunk-size.
     * More specifically, the limit * {@link #ENCRYPTION_OVERHEAD_BUFFER_FACTOR} is used to take the
     * encryption overhead into account that will increase the actual size of a chunk by approximately
     * 33%.
     *
     * <pre>Note: Large attachments will be chunked by default.</pre>
     *
     * @param attachments attachments to be chunked
     * @param attachmentLimit applicable attachment limits for a destination
     * @return list of attachments with parent attachment as key and attachment fragments as values
     * @see AttachmentChunkingConfig#getChunkSizeInMB()
     * @see Attachment#fromLargeAttachment(Path, String)
     */
    public List<AttachmentPayload> createChunkedPayloads(final List<Attachment> attachments, Limit attachmentLimit) {

        // use allowed max. limit if configured chunk-size exceeds limit
        final var maxAllowedBytes = attachmentLimit.getMaxSizeIndividualBytes();
        if (chunkSize == null || chunkSize.getSizeInBytes() > maxAllowedBytes) {
            this.chunkSize = ChunkSize.ofMB((int) (toMegaBytes(maxAllowedBytes) * ENCRYPTION_OVERHEAD_BUFFER_FACTOR));
        }

        if (config.isChunkAllAttachments()) {
            LOGGER.info("Chunking in-memory and large attachments");
            return buildChunkedPayloads(attachments);
        }
        return buildChunkedPayloadsFromLargeAttachmentsOnly(attachments);
    }

    /**
     * Transforms a list of attachments into un-chunked attachment payloads.
     *
     * @param attachments attachments to be mapped
     * @return list of attachment payloads
     */
    public List<AttachmentPayload> createPayloadsWithoutChunking(final List<Attachment> attachments) {
        return attachments.stream().map(this::buildPayloadWithoutFragments).collect(toList());
    }

    /**
     * Write fragment data to filesystem.
     *
     * @param attachmentId id of the original attachment the fragments belongs to
     * @param fragmentIndex the index of the fragment part
     * @param attachmentData the decrypted data of the attachment
     * @return a file that points to the fragment in filesystem
     */
    public File writeReceivedFragmentToFileSystem(
            final UUID attachmentId, final int fragmentIndex, final byte[] attachmentData) {
        try {
            final Path chunkedFilePath = attachmentStorageResolver.resolveIncomingAttachmentFolder(attachmentId);
            return chunker.writeFileChunk(attachmentData, fragmentIndex, chunkedFilePath);
        } catch (final IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Merge chunks for an attachment and store original file.
     *
     * @param apiAttachment attachment which fragments should be merged
     * @return optional of the merged file path
     */
    public Optional<File> mergeChunks(final ApiAttachment apiAttachment) {
        try {
            final Path path =
                    attachmentStorageResolver.resolveIncomingAttachmentFolder(apiAttachment.getAttachmentId());
            return chunker.concatChunks(path, apiAttachment.getFilename());
        } catch (final IOException e) {
            throw new RuntimeException(e);
        }
    }

    private List<AttachmentPayload> buildChunkedPayloads(final List<Attachment> attachments) {
        return attachments.stream().map(this::buildChunkedPayload).collect(toList());
    }

    private AttachmentPayload buildChunkedPayload(final Attachment attachment) {
        return attachment.isInMemoryAttachment() ? chunkInMemoryData(attachment) : chunkLargeFileData(attachment);
    }

    private List<AttachmentPayload> buildChunkedPayloadsFromLargeAttachmentsOnly(List<Attachment> attachments) {
        final var largePayloads = mapByType(attachments, Attachment::isLargeAttachment, this::chunkLargeFileData);
        final var inMemoryPayloads =
                mapByType(attachments, Attachment::isInMemoryAttachment, this::buildPayloadWithoutFragments);
        return concat(largePayloads.stream(), inMemoryPayloads.stream()).collect(toList());
    }

    private static List<AttachmentPayload> mapByType(
            List<Attachment> attachments,
            Predicate<Attachment> filter,
            Function<Attachment, AttachmentPayload> mapper) {
        return attachments.stream().filter(filter).map(mapper).collect(toList());
    }

    private AttachmentPayload chunkInMemoryData(Attachment attachment) {
        try {
            final byte[] inMemoryData = attachment.getDataAsBytes();
            LOGGER.info(
                    "Chunking in-memory data {} with a size of {}",
                    attachment.getFileName(),
                    toHumanReadableSizePrefix(inMemoryData.length));
            if (dataSizeIsLessOrEqualThanChunkSize(inMemoryData.length)) {
                return buildPayloadWithoutFragments(attachment);
            }
            final String originalDataHash = hashBytes(inMemoryData);
            final List<Fragment> attachmentFragments = buildFragmentsFromInMemoryAttachment(attachment);
            final AttachmentPayload chunkedAttachment =
                    buildFragmentedAttachmentPayload(attachment, originalDataHash, attachmentFragments);

            LOGGER.info("Split data into {} chunks", attachmentFragments.size());
            return chunkedAttachment;
        } catch (IOException e) {
            throw new FitConnectAttachmentException(e.getMessage(), e);
        }
    }

    private AttachmentPayload chunkLargeFileData(Attachment attachment) {
        try {
            final Path filePath = attachment.getLargeAttachmentFilePath();
            final long fileSize = getFileSize(filePath);
            LOGGER.info(
                    "Chunking file {} with a size of {}",
                    attachment.getFileName(),
                    toHumanReadableSizePrefix(fileSize));
            if (dataSizeIsLessOrEqualThanChunkSize(fileSize)) {
                return buildPayloadWithoutFragments(attachment);
            }
            final String originalFileHash = hashFile(filePath);
            final List<Fragment> attachmentFragments = buildFragmentsFromLargeAttachment(attachment);
            final AttachmentPayload chunkedAttachment =
                    buildFragmentedAttachmentPayload(attachment, originalFileHash, attachmentFragments);

            removeBufferedFile(attachment, filePath);

            LOGGER.info("Split file {} into {} chunks", filePath, attachmentFragments.size());
            return chunkedAttachment;
        } catch (IOException e) {
            throw new FitConnectAttachmentException(e.getMessage(), e);
        }
    }

    private List<Fragment> buildFragmentsFromLargeAttachment(final Attachment attachment) throws IOException {
        try (InputStream inputStream = Files.newInputStream(attachment.getLargeAttachmentFilePath())) {
            return chunkDataToFragments(attachment.getFileName(), getOutgoingAttachmentPath(), inputStream);
        }
    }

    private List<Fragment> buildFragmentsFromInMemoryAttachment(final Attachment attachment) throws IOException {
        try (InputStream inputStream = attachment.getDataAsInputStream()) {
            return chunkDataToFragments(attachment.getFileName(), getOutgoingAttachmentPath(), inputStream);
        }
    }

    private List<Fragment> chunkDataToFragments(String fileName, Path path, InputStream inputStream)
            throws IOException {
        return chunker.chunkStream(inputStream, chunkSize.getSizeInBytes(), fileName, path).stream()
                .map(file -> new Fragment(UUID.randomUUID(), file))
                .collect(toList());
    }

    private long getFileSize(Path dataFilePath) {
        try (FileChannel channel = FileChannel.open(dataFilePath)) {
            return channel.size();
        } catch (IOException e) {
            throw new FitConnectAttachmentException(e.getMessage(), e);
        }
    }

    private boolean dataSizeIsLessOrEqualThanChunkSize(final long dataLength) {
        if (dataLength > chunkSize.getSizeInBytes()) {
            return false;
        }
        LOGGER.info(
                "Attachment will not be chunked because its size is smaller than the chunk size of {} MB",
                chunkSize.getSizeInMB());
        return true;
    }

    private Path getOutgoingAttachmentPath() throws IOException {
        return attachmentStorageResolver.resolveOutgoingAttachmentFolder(UUID.randomUUID());
    }

    private String hashFile(final Path filePath) throws IOException {
        try (InputStream fileStream = Files.newInputStream(filePath)) {
            final byte[] hash = hashService.createHash(fileStream);
            return hashService.toHexString(hash);
        }
    }

    private String hashBytes(final byte[] data) {
        final byte[] hash = hashService.createHash(data);
        return hashService.toHexString(hash);
    }

    private void removeBufferedFile(Attachment attachment, Path filePath) {
        // only remove temp files that are used for data attachments or were buffered on disk from
        // an input stream
        if (attachment.getPurpose().equals(Purpose.DATA) || filePath.toString().contains(TEMP_BUFFERED_FILE_PREFIX)) {
            filePath.toFile().delete();
        }
    }

    private AttachmentPayload buildFragmentedAttachmentPayload(
            final Attachment attachment, final String hashedData, final List<Fragment> fragments) {
        return AttachmentPayload.builder()
                .mimeType(attachment.getMimeType())
                .fileName(attachment.getFileName())
                .description(attachment.getDescription())
                .purpose(attachment.getPurpose())
                .attachmentId(UUID.randomUUID())
                .hashedData(hashedData)
                .fragments(fragments)
                .build();
    }

    private AttachmentPayload buildPayloadWithoutFragments(final Attachment attachment) {
        return AttachmentPayload.builder()
                .mimeType(attachment.getMimeType())
                .fileName(attachment.getFileName())
                .description(attachment.getDescription())
                .purpose(attachment.getPurpose())
                .attachmentId(UUID.randomUUID())
                .data(attachment.getDataAsBytes())
                .build();
    }
}
