package dev.fitko.fitconnect.client;

import static java.util.Optional.ofNullable;

import dev.fitko.fitconnect.api.domain.zbp.attachment.ZBPApiAttachment;
import dev.fitko.fitconnect.api.domain.zbp.message.CreateMessage;
import dev.fitko.fitconnect.api.domain.zbp.message.CreateMessageResponse;
import dev.fitko.fitconnect.api.domain.zbp.state.CreateState;
import dev.fitko.fitconnect.api.domain.zbp.state.State;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectZBPException;
import dev.fitko.fitconnect.client.zbp.ZBPServiceAdapter;
import java.util.Collections;
import java.util.List;
import java.util.function.Supplier;

/**
 * ZBP client for sending messages and status updates to the ZBP (Zentrales Bürgerpostfach) of the
 * BundID.
 *
 * @see <a href="https://docs.fitko.de/fit-connect/docs/zbp">FIT-Connect und das Zentrale
 *     Bürgerpostfach (ZBP)</a>
 */
public class ZBPClient {

    private final ZBPServiceAdapter serviceAdapter;

    public ZBPClient(ZBPServiceAdapter serviceAdapter) {
        this.serviceAdapter = serviceAdapter;
    }

    /**
     * Send a new {@link CreateMessage} to a mailbox.
     *
     * @param createMessage the message payload
     * @return {@link CreateMessageResponse}
     */
    public CreateMessageResponse sendMessage(CreateMessage createMessage) {
        return wrapExceptions(
                () -> serviceAdapter.sendMessage(createMessage, Collections.emptyList(), null, null),
                "Could not send message to zbp mailbox " + createMessage.getMailboxUuid());
    }

    /**
     * Send a new {@link CreateMessage} with a list of {@link ZBPApiAttachment}s to a mailbox.
     *
     * @param createMessage the message payload incl. attachment metadata
     * @param attachments a list of attachments that contains the attachment payload
     * @return {@link CreateMessageResponse}
     */
    public CreateMessageResponse sendMessageWithAttachments(
            CreateMessage createMessage, List<ZBPApiAttachment> attachments) {
        return wrapExceptions(
                () -> serviceAdapter.sendMessage(createMessage, attachments, null, null),
                "Could not send message to zbp mailbox " + createMessage.getMailboxUuid());
    }

    /**
     * Send a new {@link CreateMessage} to a mailbox on behalf of an author.
     *
     * @param createMessage the message payload
     * @param authorCertificate the certificate of the author of the message
     * @param authorToken the JWT token of the author of the message
     * @return {@link CreateMessageResponse}
     */
    public CreateMessageResponse sendMessage(
            CreateMessage createMessage, String authorCertificate, String authorToken) {
        return wrapExceptions(
                () -> serviceAdapter.sendMessage(
                        createMessage, Collections.emptyList(), authorCertificate, authorToken),
                "Could not send message to zbp mailbox " + createMessage.getMailboxUuid());
    }

    /**
     * Send a new {@link CreateMessage} with a list of {@link ZBPApiAttachment}s to a mailbox on
     * behalf of an author.
     *
     * @param createMessage the message payload incl. attachment metadata
     * @param attachments a list of attachments that contains the attachment payload
     * @param authorCertificate the certificate of the author of the message
     * @param authorToken the JWT token of the author of the message
     * @return {@link CreateMessageResponse}
     */
    public CreateMessageResponse sendMessageWithAttachments(
            CreateMessage createMessage,
            List<ZBPApiAttachment> attachments,
            String authorCertificate,
            String authorToken) {
        return wrapExceptions(
                () -> serviceAdapter.sendMessage(createMessage, attachments, authorCertificate, authorToken),
                "Could not send message to zbp mailbox " + createMessage.getMailboxUuid());
    }

    /**
     * Send a new {@link CreateState} for an application.
     *
     * @param createState containing all state data.
     * @see State
     */
    public void createState(CreateState createState) {
        wrapExceptions(
                () -> {
                    serviceAdapter.createNewState(createState, null, null);
                    return null;
                },
                "Could not create state for application " + createState.getApplicationId());
    }

    /**
     * Send a new {@link CreateState} on behalf of an author.
     *
     * @param createState containing all state data.
     * @param authorCertificate the certificate of the author of the state to be created
     * @param authorToken the JWT token of the author of the state to be created
     * @see State
     */
    public void createState(CreateState createState, String authorCertificate, String authorToken) {
        wrapExceptions(
                () -> {
                    serviceAdapter.createNewState(createState, authorCertificate, authorToken);
                    return null;
                },
                "Could not create state for application " + createState.getApplicationId());
    }

    private <T> T wrapExceptions(final Supplier<T> supplier, final String errorMessage) {
        try {
            return supplier.get();
        } catch (final Exception e) {
            throw new FitConnectZBPException(
                    ofNullable(e.getMessage()).orElse(errorMessage),
                    ofNullable(e.getCause()).orElse(e));
        }
    }
}
