package dev.fitko.fitconnect.client;

import static java.util.Optional.ofNullable;

import dev.fitko.fitconnect.api.FitConnectService;
import dev.fitko.fitconnect.api.config.chunking.AttachmentChunkingConfig;
import dev.fitko.fitconnect.api.domain.model.attachment.Attachment;
import dev.fitko.fitconnect.api.domain.model.cases.Cases;
import dev.fitko.fitconnect.api.domain.model.destination.Destination;
import dev.fitko.fitconnect.api.domain.model.event.Event;
import dev.fitko.fitconnect.api.domain.model.event.EventLogEntry;
import dev.fitko.fitconnect.api.domain.model.event.EventPayload;
import dev.fitko.fitconnect.api.domain.model.event.Status;
import dev.fitko.fitconnect.api.domain.model.event.problems.Problem;
import dev.fitko.fitconnect.api.domain.model.metadata.Metadata;
import dev.fitko.fitconnect.api.domain.model.metadata.data.Data;
import dev.fitko.fitconnect.api.domain.model.reply.SentReply;
import dev.fitko.fitconnect.api.domain.model.submission.SentSubmission;
import dev.fitko.fitconnect.api.domain.model.submission.Submission;
import dev.fitko.fitconnect.api.domain.model.submission.SubmissionForPickup;
import dev.fitko.fitconnect.api.domain.model.submission.SubmissionsForPickup;
import dev.fitko.fitconnect.api.domain.subscriber.ReceivedSubmission;
import dev.fitko.fitconnect.api.domain.subscriber.SendableReply;
import dev.fitko.fitconnect.api.domain.validation.ValidationResult;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectAttachmentException;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectSubscriberException;
import dev.fitko.fitconnect.client.attachments.AttachmentStorageResolver;
import dev.fitko.fitconnect.client.subscriber.ReplySender;
import dev.fitko.fitconnect.client.subscriber.SubmissionReceiver;
import java.util.*;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** A subscriber-side client for retrieving submissions and sending replies. */
public class SubscriberClient {

    private static final Logger LOGGER = LoggerFactory.getLogger(SubscriberClient.class);

    private static final int DEFAULT_SUBMISSION_LIMIT = 25;
    private static final int DEFAULT_CASE_LIMIT = 500;

    private final FitConnectService fitConnectService;
    private final SubmissionReceiver submissionReceiver;
    private final ReplySender replySender;
    private final AttachmentStorageResolver attachmentStorageResolver;

    public SubscriberClient(
            final FitConnectService fitConnectService,
            final SubmissionReceiver submissionReceiver,
            final ReplySender replySender,
            AttachmentStorageResolver attachmentStorageResolver) {
        this.fitConnectService = fitConnectService;
        this.submissionReceiver = submissionReceiver;
        this.replySender = replySender;
        this.attachmentStorageResolver = attachmentStorageResolver;
    }

    /**
     * Loads a list of <strong>ALL</strong> available {@link SubmissionForPickup} that were submitted
     * to the destination.
     *
     * @param destinationId destination filter criterion for the submissions
     * @return all available submissions, retrieved in chunks of {@link
     *     SubscriberClient#DEFAULT_SUBMISSION_LIMIT}
     * @throws FitConnectSubscriberException if a technical error occurred retrieving the list of
     *     available submissions
     */
    public List<SubmissionForPickup> getAvailableSubmissionsForDestination(final UUID destinationId)
            throws FitConnectSubscriberException {
        return wrapExceptions(
                () -> collectAllAvailableSubmissions(new ArrayList<>(), destinationId, 0, DEFAULT_SUBMISSION_LIMIT),
                "Submissions could not be retrieved");
    }

    /**
     * Loads a list of available {@link SubmissionForPickup} that were submitted to the destination.
     *
     * @param destinationId filter criterion for the submissions
     * @param offset position in the dataset
     * @param limit number of submissions in result (max. is 500)
     * @return set of available submissions in the given range
     * @throws FitConnectSubscriberException if a technical error occurred retrieving the list of
     *     available submissions
     */
    public List<SubmissionForPickup> getAvailableSubmissionsForDestination(
            final UUID destinationId, final int offset, final int limit) throws FitConnectSubscriberException {
        final SubmissionsForPickup submissions = wrapExceptions(
                () -> fitConnectService.getAvailableSubmissions(destinationId, offset, limit),
                "Submissions could not be retrieved");
        LOGGER.info("Received {} submission(s) for destination {}", submissions.getCount(), destinationId);
        return submissions.getSubmissions();
    }

    /**
     * Loads a single {@link ReceivedSubmission} by object {@link SubmissionForPickup}.
     *
     * @param submission {@link SentSubmission} that is requested
     * @return {@link ReceivedSubmission} to get the submissions {@link Metadata}, {@link Data} and
     *     {@link Attachment}s as well as accept or reject the loaded submission
     * @throws FitConnectSubscriberException if a technical error occurred or validation failed
     */
    public ReceivedSubmission requestSubmission(final SubmissionForPickup submission)
            throws FitConnectSubscriberException {
        return requestSubmission(submission.getSubmissionId());
    }

    /**
     * Loads a single {@link ReceivedSubmission} by submission id.
     *
     * @param submissionId unique identifier of the requested submission
     * @return {@link ReceivedSubmission} to get the submissions {@link Metadata}, {@link Data} and
     *     {@link Attachment}s as well as accept or reject the loaded submission
     * @throws FitConnectSubscriberException if a technical error occurred or validation failed
     */
    public ReceivedSubmission requestSubmission(final UUID submissionId) throws FitConnectSubscriberException {
        return wrapExceptions(
                () -> submissionReceiver.receiveSubmission(submissionId), "Submission could not be retrieved");
    }

    /**
     * Send a reject-submission {@link Event} to confirm that the handed in submission was rejected.
     *
     * @param submissionForPickup the submission that should be rejected
     * @param rejectionProblems list of {@link Problem}s that give more detailed information on why
     *     the submission was rejected
     * @throws FitConnectSubscriberException if a technical error occurred, rejecting the submission
     * @see <a
     *     href="https://docs.fitko.de/fit-connect/docs/getting-started/event-log/events">FIT-Connect
     *     Events</a>
     */
    public void rejectSubmission(final SubmissionForPickup submissionForPickup, final List<Problem> rejectionProblems)
            throws FitConnectSubscriberException {
        wrapExceptions(
                () -> {
                    fitConnectService.rejectSubmission(
                            EventPayload.forRejectEvent(submissionForPickup, rejectionProblems));
                    return null;
                },
                "Could not reject submission");
    }

    /**
     * Checks if a received callback can be trusted by validating the provided request data
     *
     * @param hmac authentication code provided by the callback
     * @param timestampInSec timestamp in seconds provided by the callback
     * @param httpBody HTTP body provided by the callback
     * @param callbackSecret secret owned by the client, which is used to calculate the hmac
     * @return {@code true} if hmac and timestamp provided by the callback meet the required
     *     conditions
     * @throws FitConnectSubscriberException if a technical error occurred during the validation
     */
    public ValidationResult validateCallback(
            final String hmac, final Long timestampInSec, final String httpBody, final String callbackSecret)
            throws FitConnectSubscriberException {
        return wrapExceptions(
                () -> fitConnectService.validateCallback(hmac, timestampInSec, httpBody, callbackSecret),
                "Callback validation failed");
    }

    /**
     * Retrieve the current status of a {@link Submission}.
     *
     * @param destinationId unique identifier of the {@link Destination} the status should be
     *     retrieved for
     * @param caseId unique identifier of the case the status should be retrieved for
     * @param submissionId unique identifier of the submission the status should be retrieved for
     * @return {@link Status} the current status
     * @throws FitConnectSubscriberException if a technical error occurred during the status request
     */
    public Status getSubmissionStatus(final UUID destinationId, final UUID caseId, final UUID submissionId)
            throws FitConnectSubscriberException {
        return wrapExceptions(
                () -> fitConnectService.getStatus(new SentSubmission(destinationId, caseId, submissionId)),
                "Status for submission not available");
    }

    /**
     * Retrieve the event log for a given submission, case and destination.
     *
     * @param destinationId unique identifier of the {@link Destination} the log should be retrieved
     *     for
     * @param caseId unique identifier of the case the log should be retrieved for
     * @param submissionId unique identifier of the submission the log should be retrieved for
     * @return List of {@link EventLogEntry}s
     * @throws FitConnectSubscriberException if a technical or error occurred during the event-log
     *     retrieval or a validation failed
     */
    public List<EventLogEntry> getEventLogForSubmission(
            final UUID destinationId, final UUID caseId, final UUID submissionId) {
        return fitConnectService.getEventLogForSubmission(destinationId, caseId, submissionId, null);
    }

    /**
     * Retrieve the event log for a given case and destination.
     *
     * @param destinationId unique identifier of the {@link Destination} the log should be retrieved
     *     for
     * @param caseId unique identifier of the case the log should be retrieved for
     * @return List of {@link EventLogEntry}s
     * @throws FitConnectSubscriberException if a technical or error occurred during the event-log
     *     retrieval or a validation failed
     */
    public List<EventLogEntry> getEventLogForCase(final UUID destinationId, final UUID caseId) {
        return fitConnectService.getEventLogForCase(destinationId, caseId, null);
    }

    /**
     * Send a new reply for an existing submission.
     *
     * @param reply that should be sent for a submission within a case.
     * @return {@link SentReply}
     * @throws FitConnectSubscriberException if a technical error occurred during sending
     */
    public SentReply sendReply(final SendableReply reply) {
        return wrapExceptions(
                () -> replySender.sendReply(reply), "Sending reply for case " + reply.getCaseId() + " failed.");
    }

    /**
     * Retrieve the current status of a {@link SentReply}.
     *
     * @param reply sent reply the status should be retrieved for
     * @return {@link Status} the current status of the reply
     * @throws FitConnectSubscriberException if a technical error occurred during the status request
     */
    public Status getReplyStatus(final SentReply reply) throws FitConnectSubscriberException {
        return wrapExceptions(() -> fitConnectService.getStatus(reply), "Status for reply not available");
    }

    /**
     * Lists all {@link Cases} that are visible to the client (with a max. of 500).
     *
     * @return {@link Cases} list of available cases
     * @throws FitConnectSubscriberException if a technical error occurred during the request
     */
    public Cases listCases() throws FitConnectSubscriberException {
        return listCases(DEFAULT_CASE_LIMIT, 0);
    }

    /**
     * Lists all paginated {@link Cases} that are visible to the client.
     *
     * @param limit max entries per request
     * @param offset skip n elements
     * @return {@link Cases} list of available cases
     * @throws FitConnectSubscriberException if a technical error occurred during the request
     */
    public Cases listCases(final int limit, final int offset) throws FitConnectSubscriberException {
        return wrapExceptions(() -> fitConnectService.listCases(limit, offset), "Loading active cases failed.");
    }

    /**
     * Clears all persistent incoming attachments and folders from the attachment storage path,
     * configured in the {@link AttachmentChunkingConfig}.
     *
     * @throws FitConnectAttachmentException if the storage can't be cleared
     * @see AttachmentChunkingConfig#getAttachmentStoragePath()
     * @see Attachment#getLargeAttachmentFilePath()
     * @see SenderClient#clearAttachmentStorage()
     */
    public void clearAttachmentStorage() {
        wrapExceptions(
                () -> {
                    attachmentStorageResolver.clearBasePathContents();
                    return null;
                },
                "Clearing attachment storage directory failed");
    }

    private List<SubmissionForPickup> collectAllAvailableSubmissions(
            final List<SubmissionForPickup> submissions, final UUID destinationId, final int offset, final int limit) {
        final SubmissionsForPickup availableSubmissions =
                fitConnectService.getAvailableSubmissions(destinationId, offset, limit);
        submissions.addAll(availableSubmissions.getSubmissions());
        if (availableSubmissions.getTotalCount() > offset + limit) {
            return collectAllAvailableSubmissions(submissions, destinationId, offset + limit, limit);
        }
        return submissions;
    }

    private <T> T wrapExceptions(final Supplier<T> supplier, final String errorMessage) {
        try {
            return supplier.get();
        } catch (final Exception e) {
            throw new FitConnectSubscriberException(
                    ofNullable(e.getMessage()).orElse(errorMessage),
                    ofNullable(e.getCause()).orElse(e));
        }
    }
}
