package dev.fitko.fitconnect.client;

import static java.util.Optional.ofNullable;

import com.nimbusds.jose.jwk.JWK;
import dev.fitko.fitconnect.api.FitConnectService;
import dev.fitko.fitconnect.api.config.chunking.AttachmentChunkingConfig;
import dev.fitko.fitconnect.api.domain.model.cases.Cases;
import dev.fitko.fitconnect.api.domain.model.destination.Destination;
import dev.fitko.fitconnect.api.domain.model.event.Event;
import dev.fitko.fitconnect.api.domain.model.event.EventLogEntry;
import dev.fitko.fitconnect.api.domain.model.event.Status;
import dev.fitko.fitconnect.api.domain.model.event.authtags.AuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.event.problems.Problem;
import dev.fitko.fitconnect.api.domain.model.reply.RepliesForPickup;
import dev.fitko.fitconnect.api.domain.model.reply.Reply;
import dev.fitko.fitconnect.api.domain.model.submission.SentSubmission;
import dev.fitko.fitconnect.api.domain.sender.ReceivedReply;
import dev.fitko.fitconnect.api.domain.sender.SendableEncryptedSubmission;
import dev.fitko.fitconnect.api.domain.sender.SendableSubmission;
import dev.fitko.fitconnect.api.domain.validation.ValidationResult;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectAttachmentException;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectSenderException;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectSubscriberException;
import dev.fitko.fitconnect.client.attachments.AttachmentStorageResolver;
import dev.fitko.fitconnect.client.sender.ReplyReceiver;
import dev.fitko.fitconnect.client.sender.SubmissionSender;
import java.util.List;
import java.util.UUID;
import java.util.function.Supplier;

/** A sender-side client for announcing and handing in submissions and receiving replies. */
public class SenderClient {

    private static final int DEFAULT_CASE_LIMIT = 500;
    private static final int DEFAULT_REPLY_LIMIT = 500;

    private final FitConnectService fitConnectService;
    private final ReplyReceiver replyReceiver;
    private final AttachmentStorageResolver attachmentStorageResolver;
    private final SubmissionSender submissionSender;

    public SenderClient(
            final FitConnectService fitConnectService,
            final SubmissionSender submissionSender,
            final ReplyReceiver replyReceiver,
            AttachmentStorageResolver attachmentStorageResolver) {
        this.fitConnectService = fitConnectService;
        this.submissionSender = submissionSender;
        this.replyReceiver = replyReceiver;
        this.attachmentStorageResolver = attachmentStorageResolver;
    }

    /**
     * Retrieve the public encryption key for a given destination.
     *
     * @param destinationId unique identifier of a {@link Destination}
     * @return optional string containing the public JWK
     * @throws FitConnectSenderException if a technical or error occurred retrieving the key or a
     *     validation failed
     */
    public String getPublicKeyForDestination(final UUID destinationId) throws FitConnectSenderException {
        return wrapExceptions(
                () -> fitConnectService
                        .getEncryptionKeyForDestination(destinationId)
                        .toJSONString(),
                "Retrieving public key failed.");
    }

    /**
     * Retrieve the current status for a given {@link SentSubmission}.
     *
     * @param sentSubmission the original {@link SentSubmission} the status should be retrieved for
     * @return {@link Status} the current status
     * @throws FitConnectSenderException if a technical or error occurred during status retrieval or a
     *     validation failed
     */
    public Status getSubmissionStatus(final SentSubmission sentSubmission) throws FitConnectSenderException {
        return wrapExceptions(
                () -> fitConnectService.getStatus(sentSubmission), "Retrieving current status of submission failed.");
    }

    /**
     * Retrieve the full event log for a given {@link SentSubmission}.
     *
     * @param sentSubmission the original {@link SentSubmission} including {@link AuthenticationTags}
     *     that the status should be retrieved for
     * @return List of {@link EventLogEntry}s
     * @throws FitConnectSenderException if a technical or error occurred during the event-log
     *     retrieval or a validation failed
     */
    public List<EventLogEntry> getEventLogForSubmission(SentSubmission sentSubmission) {
        final UUID destinationId = sentSubmission.getDestinationId();
        final UUID caseId = sentSubmission.getCaseId();
        final UUID submissionId = sentSubmission.getSubmissionId();
        final AuthenticationTags authTags = sentSubmission.getAuthenticationTags();
        return fitConnectService.getEventLogForSubmission(destinationId, caseId, submissionId, authTags);
    }

    /**
     * Retrieve the event log for a given case and destination.
     *
     * @param destinationId unique identifier of the {@link Destination} the log should be retrieved
     *     for
     * @param caseId unique identifier of the case the log should be retrieved for
     * @return List of {@link EventLogEntry}s
     * @throws FitConnectSenderException if a technical or error occurred during the event-log
     *     retrieval or a validation failed
     */
    public List<EventLogEntry> getEventLogForCase(final UUID destinationId, final UUID caseId) {
        return fitConnectService.getEventLogForCase(destinationId, caseId, null);
    }

    /**
     * Checks if a received callback can be trusted by validating the provided request data.
     *
     * @param hmac authentication code provided by the callback
     * @param timestampInSec timestamp in seconds provided by the callback
     * @param httpBody HTTP body provided by the callback
     * @param callbackSecret secret owned by the client, which is used to calculate the hmac
     * @return {@code ValidationResult.ok()} if hmac and timestamp provided by the callback meet the
     *     required conditions
     * @throws FitConnectSenderException if a technical error occurred during the validation
     */
    public ValidationResult validateCallback(
            final String hmac, final Long timestampInSec, final String httpBody, final String callbackSecret)
            throws FitConnectSenderException {
        return wrapExceptions(
                () -> fitConnectService.validateCallback(hmac, timestampInSec, httpBody, callbackSecret),
                "Callback validation could not be applied.");
    }

    /**
     * Send submission with attachments and data to FIT-Connect API. Encryption and validation is
     * handled by the SDK.
     *
     * @return {@link SentSubmission} the handed in submission response
     * @throws FitConnectSenderException if a technical error occurred during sending or a validation
     *     failed
     */
    public SentSubmission send(final SendableSubmission sendableSubmission) throws FitConnectSenderException {
        return wrapExceptions(() -> submissionSender.sendSubmission(sendableSubmission), "Sending submission failed.");
    }

    /**
     * Send an already encrypted submission to FIT-Connect API.
     *
     * @return {@link SentSubmission} the handed in submission response
     * @throws FitConnectSenderException if a technical error occurred during sending or a validation
     *     failed
     */
    public SentSubmission send(final SendableEncryptedSubmission sendableEncryptedSubmission)
            throws FitConnectSenderException {
        return wrapExceptions(
                () -> submissionSender.sendEncryptedSubmission(sendableEncryptedSubmission),
                "Sending already encrypted submission failed.");
    }

    /**
     * Get {@link Reply} for a submission.
     *
     * @param replyId unique identifier of the reply
     * @param privateReplyDecryptionKey the decryption key as JWK
     * @return reply with encrypted metadata and data
     */
    public ReceivedReply getReply(final UUID replyId, final JWK privateReplyDecryptionKey) {
        return wrapExceptions(
                () -> replyReceiver.receiveReply(replyId, privateReplyDecryptionKey.toRSAKey()),
                "Loading reply failed.");
    }

    /**
     * Send a reject-reply {@link Event} to confirm that the handed in reply was rejected.
     *
     * @param replyId the replyId that should be rejected
     * @param rejectionProblems list of {@link Problem}s that give more detailed information on why
     *     the reply was rejected
     * @throws FitConnectSubscriberException if a technical error occurred rejecting the submission
     * @see <a
     *     href="https://docs.fitko.de/fit-connect/docs/getting-started/event-log/events">FIT-Connect
     *     Events</a>
     */
    public void rejectReply(final UUID replyId, final List<Problem> rejectionProblems)
            throws FitConnectSubscriberException {
        wrapExceptions(
                () -> {
                    fitConnectService.rejectReply(replyId, rejectionProblems);
                    return null;
                },
                "Could not reject reply");
    }

    /**
     * Lists all {@link Cases} that are visible to the client (with a max. of 500).
     *
     * @return {@link Cases} list of available cases
     * @throws FitConnectSenderException if a technical error occurred during the request
     */
    public Cases listCases() throws FitConnectSenderException {
        return listCases(0, DEFAULT_CASE_LIMIT);
    }

    /**
     * Lists all {@link Cases} that are visible to the client.
     *
     * @param limit max entries per request
     * @param offset skip n elements
     * @return {@link Cases} list of available cases
     * @throws FitConnectSenderException if a technical error occurred during the request
     */
    public Cases listCases(final int limit, final int offset) throws FitConnectSenderException {
        return wrapExceptions(() -> fitConnectService.listCases(limit, offset), "Loading active cases failed.");
    }

    /**
     * Lists all {@link RepliesForPickup} that are visible to the client (with a max. of 500).
     *
     * @return {@link RepliesForPickup} list of available replies ready for pickup
     * @throws FitConnectSenderException if a technical error occurred during the request
     */
    public RepliesForPickup getAvailableReplies() {
        return wrapExceptions(
                () -> fitConnectService.getAvailableReplies(DEFAULT_REPLY_LIMIT, 0), "Loading replies failed");
    }

    /**
     * Lists all {@link RepliesForPickup} that are visible to the client.
     *
     * @param limit max entries per request
     * @param offset skip n elements
     * @return {@link RepliesForPickup} list of available replies ready for pickup
     * @throws FitConnectSenderException if a technical error occurred during the request
     */
    public RepliesForPickup getAvailableReplies(final int limit, final int offset) {
        return wrapExceptions(() -> fitConnectService.getAvailableReplies(limit, offset), "Loading replies failed");
    }

    /**
     * Clears all persistent incoming attachments and folders from the attachment storage path,
     * configured in the {@link AttachmentChunkingConfig}.
     *
     * @throws FitConnectAttachmentException if the storage cannot be cleared
     * @see AttachmentChunkingConfig#getAttachmentStoragePath()
     * @see SubscriberClient#clearAttachmentStorage()
     */
    public void clearAttachmentStorage() {
        wrapExceptions(
                () -> {
                    attachmentStorageResolver.clearBasePathContents();
                    return null;
                },
                "Clearing attachment storage directory failed");
    }

    private <T> T wrapExceptions(final Supplier<T> supplier, final String errorMessage) {
        try {
            return supplier.get();
        } catch (final Exception e) {
            throw new FitConnectSenderException(
                    ofNullable(e.getMessage()).orElse(errorMessage),
                    ofNullable(e.getCause()).orElse(e));
        }
    }
}
