package dev.fitko.fitconnect.client;

import static java.util.Optional.ofNullable;

import dev.fitko.fitconnect.api.domain.model.route.Area;
import dev.fitko.fitconnect.api.domain.model.route.Route;
import dev.fitko.fitconnect.api.domain.model.route.RouteResult;
import dev.fitko.fitconnect.api.domain.routing.DestinationSearch;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectRouterException;
import dev.fitko.fitconnect.api.services.routing.RoutingService;
import dev.fitko.fitconnect.core.routing.RouteVerifier;
import java.util.List;
import java.util.function.Supplier;

/** A client to retrieve routing information for areas and destinations. */
public final class RouterClient {

    private final RoutingService routingService;
    private final RouteVerifier routeVerifier;

    public RouterClient(final RoutingService routingService, final RouteVerifier routeVerifier) {
        this.routingService = routingService;
        this.routeVerifier = routeVerifier;
    }

    /**
     * Finds a list of {@link Route}s by a given service identifier and an area search criterion.
     * Invalid routes are automatically filtered out and validation errors are logged.
     *
     * @param search search parameters that contain leikaKey and one other area search criterion see
     *     {@link DestinationSearch}
     * @return list of found and validated {@link Route}s matching the search criteria
     * @throws FitConnectRouterException if a technical error occurred during the routing operation
     */
    public List<Route> findDestinations(final DestinationSearch search) throws FitConnectRouterException {

        var leikaKey = search.getLeikaKey();
        var ars = search.getArs();
        var ags = search.getAgs();
        var areaId = search.getAreaId();
        var offset = search.getOffset();
        var limit = search.getLimit();

        final RouteResult routeResult = wrapExceptions(
                () -> routingService.getRoutes(leikaKey, ars, ags, areaId, offset, limit),
                "Finding destination failed");

        return routeVerifier.validateRouteDestinations(routeResult.getRoutes(), leikaKey, ars);
    }

    /**
     * Finds a list of {@link Area}s based on a filter criterion, e.g. a zip code or city.
     *
     * @param filter filter criterion as string
     * @param offset offset to start from
     * @param limit max entries
     * @return list of {@link Area}
     * @throws FitConnectRouterException if a technical error occurred during the query
     */
    public List<Area> findAreas(final String filter, final int offset, final int limit)
            throws FitConnectRouterException {
        return wrapExceptions(() -> findAreas(List.of(filter), offset, limit), "Finding areas failed");
    }

    /**
     * Find a list {@link Area}s based on a list of multiple filter criteria, e.g. a zip code or a
     * city as in List.of("01234", "City*").
     *
     * @param filters list of string filters
     * @param offset offset to start from
     * @param limit max entries
     * @return list of {@link Area}
     * @throws FitConnectRouterException if a technical error occurred during the query
     */
    public List<Area> findAreas(final List<String> filters, final int offset, final int limit)
            throws FitConnectRouterException {
        return wrapExceptions(
                () -> routingService.getAreas(filters, offset, limit).getAreas(), "Finding areas failed");
    }

    private <T> T wrapExceptions(final Supplier<T> supplier, final String errorMessage) {
        try {
            return supplier.get();
        } catch (final Exception e) {
            throw new FitConnectRouterException(
                    ofNullable(e.getMessage()).orElse(errorMessage),
                    ofNullable(e.getCause()).orElse(e));
        }
    }
}
