package dev.fitko.fitconnect.client;

import static java.util.Optional.ofNullable;

import dev.fitko.fitconnect.api.domain.limits.Limit;
import dev.fitko.fitconnect.api.domain.limits.destination.ChangeRequestStatus;
import dev.fitko.fitconnect.api.domain.limits.destination.DestinationLimits;
import dev.fitko.fitconnect.api.domain.limits.destination.LimitChangeRequest;
import dev.fitko.fitconnect.api.domain.model.destination.CreateDestination;
import dev.fitko.fitconnect.api.domain.model.destination.Destination;
import dev.fitko.fitconnect.api.domain.model.destination.Destinations;
import dev.fitko.fitconnect.api.domain.model.jwk.ApiJwk;
import dev.fitko.fitconnect.api.domain.model.jwk.ApiJwks;
import dev.fitko.fitconnect.api.exceptions.client.FitConnectDestinationException;
import dev.fitko.fitconnect.api.services.destination.DestinationService;
import java.util.UUID;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A client to manage destinations and their public-keys via CRUD operations. <br>
 * <br>
 * The used client-credentials need the following OAuth-Scope to manage a destination:
 * <i>https://schema.fitko.de/fit-connect/oauth/scopes/manage-destinations</i> <br>
 * <br>
 * Check your client-setup created in the <a
 * href="https://portal.auth-testing.fit-connect.fitko.dev">Self-Service-Portal (TEST)</a>
 *
 * @see <a
 *     href="https://docs.fitko.de/fit-connect/docs/receiving/destination-management">Zustellpunkte
 *     Verwalten</a>
 * @see <a
 *     href="https://docs.fitko.de/fit-connect/docs/details/authentication/scopes-subscriber#scopes-f%C3%BCr-empfangende-systeme">OAuth-Scopes
 *     Docs</a>
 * @see <a href="https://docs.fitko.de/fit-connect/docs/apis/destination-api/">Destination-API</a>
 */
public class DestinationClient {

    private static final Logger LOGGER = LoggerFactory.getLogger(DestinationClient.class);

    private final DestinationService destinationService;

    public DestinationClient(DestinationService destinationService) {
        this.destinationService = destinationService;
    }

    /**
     * Gets a {@link Destination} by its id.
     *
     * @param destinationId unique identifier of the destination
     * @return the destination
     * @throws FitConnectDestinationException if the destination couldn't be retrieved or a technical
     *     error occurred
     */
    public Destination getDestination(UUID destinationId) {
        LOGGER.info("Getting destination {}", destinationId);
        return wrapExceptions(() -> destinationService.getDestination(destinationId), "Getting destination failed");
    }

    /**
     * Create a new destination.
     *
     * @param destination {@link CreateDestination} payload
     * @return the {@link Destination} with a generated destinationId
     * @throws FitConnectDestinationException if the destination couldn't be created or a technical
     *     error occurred
     */
    public Destination createDestination(CreateDestination destination) {
        LOGGER.info("Creating new destination");
        return wrapExceptions(
                () -> destinationService.createDestination(destination), "Creating new destination failed");
    }

    /**
     * Update a given destination.
     *
     * <p>All fields of the destination will be updated and overwritten.
     *
     * <p>An example would be changing the encryptionKid field during key rollover. The workflow is as
     * follows:
     *
     * <ol>
     *   <li>Load the destination that should be updated via {@link #getDestination(UUID)}
     *   <li>Change fields to their new value
     *   <li>Send updated destination (internally a PUT-request is executed)
     * </ol>
     *
     * <p>When overwriting destinations in the 'draft' status, it should be noted that the sub-objects
     * (services, contactInformation, etc.) must be internally valid.
     *
     * @param destination the destination with fields to be updated.
     * @return the updated {@link Destination}
     * @throws FitConnectDestinationException if the destination couldn't be updated or a technical
     *     error occurred
     */
    public Destination updateDestination(Destination destination) {
        LOGGER.info("Updating destination {}", destination.getDestinationId());
        return wrapExceptions(() -> destinationService.updateDestination(destination), "Updating destination failed");
    }

    /**
     * List all self-created destinations and their configurations.
     *
     * @param offset position in the dataset
     * @param limit number of destinations in the result-set (max. is 500)
     * @return {@link Destinations} object with a set of destinations
     * @throws FitConnectDestinationException if the destinations couldn't be retrieved or a technical
     *     error occurred
     */
    public Destinations listDestinations(final int offset, final int limit) {
        LOGGER.info("Listing available destinations");
        return wrapExceptions(() -> destinationService.listDestinations(offset, limit), "Listing destinations failed");
    }

    /**
     * Delete a destination.
     *
     * <p>Removal is allowed as long as it is in the 'created' status.
     *
     * @param destinationId unique identifier of the destination to be deleted
     * @throws FitConnectDestinationException if the destination couldn't be deleted or a technical
     *     error occurred
     */
    public void deleteDestination(UUID destinationId) {
        LOGGER.info("Deleting destination {}", destinationId);
        wrapExceptions(
                () -> {
                    destinationService.deleteDestination(destinationId);
                    return null;
                },
                "Deleting destination failed");
    }

    /**
     * Retrieve a public-key of a given destination. The key is in JWKS RFC7517 format.
     *
     * @param destinationId unique identifier of the destination to be deleted
     * @param keyId unique identifier of a destinations public key
     * @return the public-key as {@link ApiJwk}
     * @throws FitConnectDestinationException if the key couldn't be retrieved or a technical error
     *     occurred
     */
    public ApiJwk getKeyForDestination(UUID destinationId, String keyId) {
        LOGGER.info("Loading key {} for destination {}", keyId, destinationId);
        return wrapExceptions(() -> destinationService.getKey(destinationId, keyId), "Loading key failed");
    }

    /**
     * List all public keys associated with a given destination. The keys are in JWKS RFC7517 format.
     *
     * @param destinationId unique identifier of the destination
     * @param offset position in the dataset
     * @param limit number of destinations in the result-set (max. is 500)
     * @return {@link ApiJwks} list of JWKs
     * @throws FitConnectDestinationException if the keys couldn't be retrieved or a technical error
     *     occurred
     */
    public ApiJwks getKeysForDestination(UUID destinationId, final int offset, final int limit) {
        LOGGER.info("Loading available keys for destination {}", destinationId);
        return wrapExceptions(() -> destinationService.listKeys(destinationId, offset, limit), "Loading keys failed");
    }

    /**
     * Adds a new JWK to the destination.
     *
     * <p>Be aware to NOT send private keys to the destination.
     *
     * @param destinationId unique identifier of the destination
     * @param publicKey the public key for encryption or signature verification as {@link ApiJwk}
     * @throws FitConnectDestinationException if the key couldn't be added or a technical error
     *     occurred
     */
    public void addKeyToDestination(UUID destinationId, ApiJwk publicKey) {
        LOGGER.info("Adding key {} to destination {}", publicKey.getKid(), destinationId);
        wrapExceptions(
                () -> {
                    destinationService.addKey(destinationId, publicKey);
                    return null;
                },
                "Adding new key to destination failed");
    }

    /**
     * Get attachment limits that apply to sending submissions and replies to this destination.
     *
     * @param destinationId unique identifier of the destination
     * @return the {@link DestinationLimits} for the requested destination
     * @throws FitConnectDestinationException if the limits aren't available or a technical error
     *     occurred
     */
    public DestinationLimits getAttachmentLimits(UUID destinationId) {
        LOGGER.info("Loading attachment limits for destination {}", destinationId);
        return wrapExceptions(
                () -> destinationService.getAttachmentLimits(destinationId), "Loading attachment limits failed");
    }

    /**
     * Request a limit change for a given destination with {@link ChangeRequestStatus#PENDING}.
     *
     * <p>A new request always overwrites an existing open request.
     *
     * @param destinationId unique identifier of a destination
     * @param limitValues the new limit values that are requested
     * @param requestReason a brief explanation of why an increase of the current limits is desired
     * @param contactEmail mail-address of a contact person who would like to be informed about the
     *     result
     * @return changed {@link DestinationLimits} for this destination
     */
    public DestinationLimits requestLimitChange(
            UUID destinationId, Limit limitValues, String requestReason, String contactEmail) {
        LOGGER.info("Requesting limit change for destination {}", destinationId);
        final LimitChangeRequest request = LimitChangeRequest.builder()
                .status(ChangeRequestStatus.PENDING)
                .requestReason(requestReason)
                .contactEmail(contactEmail)
                .values(limitValues)
                .build();
        return wrapExceptions(
                () -> destinationService.requestLimitChange(destinationId, request), "Requesting limit change failed");
    }

    /**
     * Request a limit change for a given destination with {@link ChangeRequestStatus#PENDING}.
     *
     * <p>A new request always overwrites an existing open request.
     *
     * @param destinationId unique identifier of a destination
     * @param limitValues the new limit values that are requested
     * @param requestReason a brief explanation of why an increase of the current limits is desired
     * @return changed {@link DestinationLimits} for this destination
     */
    public DestinationLimits requestLimitChange(UUID destinationId, Limit limitValues, String requestReason) {
        return requestLimitChange(destinationId, limitValues, requestReason, null);
    }

    /**
     * Revoke a currently pending limit change request for a given destination with {@link
     * ChangeRequestStatus#WITHDRAWN}.
     *
     * @param destinationId unique identifier of a destination
     * @return current {@link DestinationLimits} for this destination
     */
    public DestinationLimits withdrawLimitChangeRequest(UUID destinationId) {
        LOGGER.info("Withdrawing current limit change request for destination {}", destinationId);
        final LimitChangeRequest request = LimitChangeRequest.builder()
                .status(ChangeRequestStatus.WITHDRAWN)
                .build();
        return wrapExceptions(
                () -> destinationService.requestLimitChange(destinationId, request),
                "Withdrawing change request failed");
    }

    private <T> T wrapExceptions(final Supplier<T> supplier, final String errorMessage) {
        try {
            return supplier.get();
        } catch (final Exception e) {
            throw new FitConnectDestinationException(
                    ofNullable(e.getMessage()).orElse(errorMessage),
                    ofNullable(e.getCause()).orElse(e));
        }
    }
}
