package dev.fitko.fitconnect.api.services.validation;

import dev.fitko.fitconnect.api.domain.validation.VirusScanEngineInfo;
import dev.fitko.fitconnect.api.domain.validation.VirusScanResult;
import dev.fitko.fitconnect.api.exceptions.internal.VirusScanException;
import java.io.InputStream;
import java.nio.file.Path;

/**
 * A service that provides virus scanning capabilities for files and data streams. Implementations
 * can use different virus scanning engines (for example, ClamAV or other commercial solutions).
 */
public interface VirusScanService {

    /**
     * Scans a byte array for viruses.
     *
     * @param data the data to scan
     * @return {@link VirusScanResult} containing the scan results
     * @throws VirusScanException if the scan fails or the scanning service is unavailable
     */
    VirusScanResult scanBytes(byte[] data) throws VirusScanException;

    /**
     * Scans an input stream for viruses.
     *
     * @param inputStream the input stream to scan
     * @return {@link VirusScanResult} containing the scan results
     * @throws VirusScanException if the scan fails or the scanning service is unavailable
     */
    VirusScanResult scanStream(InputStream inputStream) throws VirusScanException;

    /**
     * Scans a file at the specified path for viruses.
     *
     * @param filePath the path to the file to scan
     * @return {@link VirusScanResult} containing the scan results
     * @throws VirusScanException if the scan fails, the file doesn't exist, or the scanning service
     *     is unavailable
     */
    VirusScanResult scanFile(Path filePath) throws VirusScanException;

    /**
     * Checks if the virus scanning service is available.
     *
     * @return true if the service is available, false otherwise
     */
    boolean isAvailable();

    /**
     * Gets information about the virus scanning engine.
     *
     * @return {@link VirusScanEngineInfo} containing engine information
     */
    VirusScanEngineInfo getEngineInfo();
}
