package dev.fitko.fitconnect.api.services.validation;

import com.nimbusds.jose.jwk.KeyOperation;
import com.nimbusds.jose.jwk.RSAKey;
import dev.fitko.fitconnect.api.domain.model.destination.PublicDestination;
import dev.fitko.fitconnect.api.domain.model.event.Event;
import dev.fitko.fitconnect.api.domain.model.event.authtags.AuthenticationTags;
import dev.fitko.fitconnect.api.domain.model.metadata.Metadata;
import dev.fitko.fitconnect.api.domain.model.metadata.attachment.AttachmentForValidation;
import dev.fitko.fitconnect.api.domain.model.metadata.data.Data;
import dev.fitko.fitconnect.api.domain.model.reply.Reply;
import dev.fitko.fitconnect.api.domain.model.submission.Submission;
import dev.fitko.fitconnect.api.domain.validation.ValidationResult;
import java.io.InputStream;
import java.net.URI;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * Validator for publicKeys and metadata.
 *
 * @see <a href="https://docs.fitko.de/fit-connect/docs/sending/encrypt/#certificateValidation">Certificate Validation</a>
 */
public interface ValidationService {

    /**
     * Validates the public key for integrity.
     *
     * @param publicKey    the public JWK
     * @param keyOperation key operation the public key be validated with, represents {@code key_ops} parameter in a JWK
     * @return {@link ValidationResult} that includes an error if the validation failed
     */
    ValidationResult validatePublicKey(RSAKey publicKey, KeyOperation keyOperation);

    /**
     * Validates the public key for integrity.
     *
     * @param publicKey      the public JWK
     * @param validationDate date for which the validity of the certificate chain (x5c) should be determined
     * @param keyOperation   key operation the public key be validated with, represents {@code key_ops} parameter in a JWK
     * @return {@link ValidationResult} that includes an error if the validation failed
     */
    ValidationResult validatePublicKey(RSAKey publicKey, Date validationDate, KeyOperation keyOperation);

    /**
     * Validates the metadata against a given schema.
     *
     * @param metadata the metadata object that should be validated
     * @return a {@link ValidationResult} with an optional error
     */
    ValidationResult validateMetadataSchema(Metadata metadata);

    /**
     * Validates the submissions {@link Metadata} structure and contents to ensure its correctness.
     *
     * @param metadata                the {@link Metadata} object that is validated
     * @param submission              the {@link Submission} of the validated metadata
     * @param destination             the {@link PublicDestination} of the validated metadata
     * @param eventAuthenticationTags the {@link AuthenticationTags} of the submit event of the submission
     * @return a {@link ValidationResult}, contains an error if the {@link Metadata} is invalid, e.g. doesn't match the schema
     */
    ValidationResult validateSubmissionMetadata(
            Metadata metadata,
            Submission submission,
            PublicDestination destination,
            AuthenticationTags eventAuthenticationTags);

    /**
     * Validates the replies {@link Metadata} structure and contents to ensure its correctness.
     *
     * @param metadata                the {@link Metadata} object that is validated
     * @param reply                   the {@link Reply} of the validated metadata
     * @param destination             the {@link PublicDestination} of the validated metadata
     * @param eventAuthenticationTags the {@link AuthenticationTags} of the submit event of the reply
     * @return a {@link ValidationResult}, contains an error if the {@link Metadata} is invalid, e.g. doesn't match the schema
     */
    ValidationResult validateReplyMetadata(
            Metadata metadata, Reply reply, PublicDestination destination, AuthenticationTags eventAuthenticationTags);

    /**
     * Validates the attachment structure and contents to ensure its correctness.
     *
     * @param attachmentsForValidation list of attachments containing the hash, decrypted and encrypted data needed for validation
     * @param authenticationTags       the {@link AuthenticationTags} of the validated attachments
     * @return a {@link ValidationResult}, contains an error if the attachment is invalid
     */
    ValidationResult validateAttachments(
            List<AttachmentForValidation> attachmentsForValidation, AuthenticationTags authenticationTags);

    /**
     * Validates the {@link Data} structure and contents to ensure its correctness.
     *
     * @param decryptedData          the unencrypted data as byte[]
     * @param encryptedData          the encrypted data of the {@link Submission}
     * @param metadata               the {@link Metadata} of the validated data
     * @param eventAuthenticationTag the authenticationTag of the data from an {@link Event}
     * @return a {@link ValidationResult}, contains an error if the {@link Metadata} is invalid, e.g. doesn't match the schema
     */
    ValidationResult validateData(
            byte[] decryptedData, String encryptedData, Metadata metadata, String eventAuthenticationTag);

    /**
     * Validates a set event against a given schema.
     *
     * @param setEventPayload the set event that should be validated
     * @return a {@link ValidationResult} with an optional error
     */
    ValidationResult validateSetEventSchema(String setEventPayload);

    /**
     * Validates signature payload claims against a given schema.
     *
     * @param destinationPayload the payload to be validated
     * @return a {@link ValidationResult} with an optional error
     */
    ValidationResult validateDestinationSchema(Map<String, Object> destinationPayload);

    /**
     * Compares a given byte[] to its original hash value.
     *
     * @param originalHexHash hash of the data as hex string
     * @param data            byte[] of data that will be compared to the original hash
     * @return a {@link ValidationResult} with an optional error
     */
    ValidationResult validateHashIntegrity(String originalHexHash, byte[] data);

    /**
     * Compares a given input-stream to its original hash value.
     *
     * @param originalHexHash hash of the data as hex string
     * @param inputStream     input-stream of the data that will be compared to the original hash
     * @return a {@link ValidationResult} with an optional error
     */
    ValidationResult validateHashIntegrity(String originalHexHash, InputStream inputStream);

    /**
     * Tests if given JSON data matches the specified schema.
     *
     * @param json      JSON byte[] that is tested
     * @param schemaUri URI of schema to validate against
     * @return a {@link ValidationResult} with an optional error
     */
    ValidationResult validateJsonSubmissionDataSchema(byte[] json, URI schemaUri);

    /**
     * Tests if given XML data matches the specified schema.
     *
     * @param xml       XML byte[] that is tested
     * @param schemaUri URI of schema to validate against
     * @return a {@link ValidationResult} with an optional error
     */
    ValidationResult validateXmlSubmissionDataSchema(byte[] xml, URI schemaUri);

    /**
     * Tests if a given byte[] is well-formed JSON syntax.
     *
     * @param json json as byte[] that is tested
     * @return a {@link ValidationResult} with an optional error
     */
    ValidationResult validateJsonFormat(byte[] json);

    /**
     * Tests if a given byte[] is well-formed XML syntax.
     *
     * @param xml xml as byte[] that is tested
     * @return a {@link ValidationResult} with an optional error
     */
    ValidationResult validateXmlFormat(byte[] xml);

    /**
     * Checks if a received callback can be trusted by validating the provided request data
     *
     * @param hmac               authentication code provided by the callback
     * @param timestampInSeconds timestamp in seconds provided by the callback
     * @param httpBody           HTTP body provided by the callback
     * @param callbackSecret     secret owned by the client, which is used to calculate the hmac
     * @return {@code true} if hmac and timestamp provided by the callback meet the required conditions
     */
    ValidationResult validateCallback(String hmac, Long timestampInSeconds, String httpBody, String callbackSecret);
}
